<?php
namespace Mvc;

class Feature
{
    const IMAGE_FEATURES_FILE = '/etc/image_features';

    private $configs = array();

    protected function __construct()
    {
        // load entire config file
        $lines = file(self::IMAGE_FEATURES_FILE);

        // process each config file line
        foreach ($lines as $line) {
            // split on the "=" character
            $parts = preg_split('/=/', trim($line), 2);

            // if we have 2 parts, the first part is the key and the second is the value
            if (count($parts) == 2) {
                $this->configs[$parts[0]] = $parts[1];
            }
        }
    }

    /**
     * @param $key - config key (ex: FEATURE_CLI, HW_TYPE, PRODUCT_CLASS, etc)
     *
     * @return string|null - value set in config file, null if not set
     */
    protected function get($key)
    {
        return isset($this->configs[$key]) ? $this->configs[$key] : null;
    }

    /**
     * @param $feature - feature name (ex: 3PCC, ACD, CLI, DYNDNS, E1T1, etc)
     *
     * @return bool - true if product supports feature, false if product does not support feature
     */
    protected function getBoolFeature($feature)
    {
        return (boolean) $this->get('FEATURE_' . strtoupper($feature));
    }

    protected function getFeature($feature)
    {
        return $this->get($feature);
    }

    /**
     * Treat the feature class as a singleton.  We create just 1 instance and reuse references to that instance.
     * @return Feature - instance of the feature config class
     */
    protected static function getInstance()
    {
        static $obj = null;
        return $obj ? : $obj = new self();
    }

    public static function getHwVersion()
    {
        return self::getInstance()->getFeature('HW_VERSION');
    }

    public static function has3pcc()
    {
        return self::getInstance()->getBoolFeature('3PCC');
    }

    public static function hasAcd()
    {
		if ( self::isqx20 () )
		{
	  	  return false;
		}
        return self::getInstance()->getBoolFeature('ACD');
    }

    public static function hasCrmConnect()
    {
        return self::getInstance()->getBoolFeature('CRM_CONNECT');
    }

    public static function hasWAMP()
    {
        return self::getInstance()->getBoolFeature('WAMP');
    }

    public static function hasAudioLines()
    {
        return self::getInstance()->getBoolFeature('AUDIO_LINES');
    }

    public static function hasBargeIn()
    {
    	if ( self::isqx20 () )
		{
	  	  return false;
		}

        return self::getInstance()->getBoolFeature('BARGEIN');
    }

    public static function hasCdrArchive()
    {
        return self::getInstance()->getBoolFeature('CDR_ARCHIVE');
    }

    public static function hasConfServer()
    {
        return self::getInstance()->getBoolFeature('CONF_SERVER');
    }

    public static function hasDhcpClient()
    {
        return self::getInstance()->getBoolFeature('DHCPCLIENT');
    }

    public static function hasDnsServer()
    {
        return self::getInstance()->getBoolFeature('DNSSERVER');
    }

    public static function hasDynDns()
    {
        return self::getInstance()->getBoolFeature('DYNDNS');
    }

    public static function hasE1T1()
    {
        return self::getInstance()->getBoolFeature('E1T1');
    }

    public static function hasExtensionBulk()
    {
        return self::getInstance()->getBoolFeature('EXTENSION_BULK');
    }

    public static function hasFeature()
    {
        return self::getInstance()->getBoolFeature('FEATURE');
    }

    public static function hasFirewall()
    {
        return self::getInstance()->getBoolFeature('FIREWALL');
    }

    public static function hasFxo()
    {
        return self::getInstance()->getBoolFeature('FXO');
    }

    public static function hasFxs()
    {
        return self::getInstance()->getBoolFeature('FXS');
    }

    public static function hasIds()
    {
        return self::getInstance()->getBoolFeature('IDS');
    }

    public static function hasIpLines()
    {
        return self::getInstance()->getBoolFeature('IP_LINES');
    }

    public static function hasIpLinesAutoConfig()
    {
        return self::getInstance()->getBoolFeature('IPLINES_AUTOCONFIG');
    }

    public static function hasIsdn()
    {
        return self::getInstance()->getBoolFeature('ISDN');
    }

    public static function hasIvrBuilder()
    {
        return self::getInstance()->getBoolFeature('IVR_BUILDER');
    }

    public static function hasKeySystemEmulation()
    {
        return self::getInstance()->getBoolFeature('KEY_SYSTEM_EMULATION');
    }

    public static function hasLogArchive()
    {
        return self::getInstance()->getBoolFeature('LOG_ARCHIVE');
    }

    public static function hasLongUnitId()
    {
        return self::getInstance()->getBoolFeature('LONG_UNIT_ID');
    }

    public static function hasNatRouter()
    {
        return self::getInstance()->getBoolFeature('NAT_ROUTER');
    }

    public static function hasPmsSupport()
    {
        return self::getInstance()->getBoolFeature('PMS_SUPPORT');
    }

    public static function hasPnpIpPhones()
    {
        return self::getInstance()->getBoolFeature('PHPIPPHONES');
    }

    public static function hasPpp()
    {
        return self::getInstance()->getBoolFeature('PPP');
    }

    public static function hasPresence()
    {
        return self::getInstance()->getBoolFeature('PRESENCE');
    }

    public static function hasProvisioning()
    {
        return self::getInstance()->getBoolFeature('PROVISIONING');
    }

    public static function hasPstnExAutoConfig()
    {
        return self::getInstance()->getBoolFeature('PSTN_EX_AUTOCONFIG');
    }

    public static function hasRadiusClient()
    {
        return self::getInstance()->getBoolFeature('RADIUS_CLIENT');
    }

    public static function hasRecording()
    {
        return self::getInstance()->getBoolFeature('RECORDING');
    }

    public static function hasRedundancy()
    {
        return self::getInstance()->getBoolFeature('REDUNDANCY');
    }

    public static function hasRelease()
    {
        return self::getInstance()->getBoolFeature('RELEASE');
    }

    public static function hasRemoteControl()
    {
        return self::getInstance()->getBoolFeature('REMOTE_CONTROL');
    }

    public static function hasRemoteTesting()
    {
        return self::getInstance()->getBoolFeature('REMOTE_TESTING');
    }

    public static function hasRoutingServer()
    {
        return self::getInstance()->getBoolFeature('ROUTING_SERVER');
    }

    public static function hasSipLines()
    {
        return self::getInstance()->getBoolFeature('SIP_LINES');
    }

    public static function hasSipTrunk()
    {
        return self::getInstance()->getBoolFeature('SIP_TRUNK');
    }

    public static function hasSms()
    {
        return self::getInstance()->getBoolFeature('SMS');
    }

    public static function hasSqlServer()
    {
        return self::getInstance()->getBoolFeature('SQLSERVER');
    }

    public static function hasSnmp()
    {
        return self::getInstance()->getBoolFeature('SNMP');
    }

    public static function hasTlsSupport()
    {
        return self::getInstance()->getBoolFeature('TLS_SUPPORT');
    }

    public static function hasVlan()
    {
        return self::getInstance()->getBoolFeature('VLAN');
    }

    public static function hasVoiceMailSystem()
    {
        return self::getInstance()->getBoolFeature('VMS');
    }

    public static function hasVpn()
    {
        return self::getInstance()->getBoolFeature('VPN');
    }

    public static function hasTelephony()
    {
        return self::getInstance()->getBoolFeature('TELEPHONY');
    }

    public static function hasWakeOnLan()
    {
        return self::getInstance()->getBoolFeature('WAKE_ON_LAN');
    }
	
    public static function hasSQLITE()
    {
        return self::getInstance()->getBoolFeature('SQLITE');
    }

    public static function hasOpenVpn()
    {
        return self::getInstance()->getBoolFeature('OPENVPN');
    }

    public static function hasOpenVpnClient()
    {
        return self::getInstance()->getBoolFeature('OPENVPN_CLIENT');
    }

    public static function hasRtsp()
    {
        return self::getInstance()->getBoolFeature('RTSP');
    }

    public static function hasCloudOwnService()
    {
        return self::getInstance()->getBoolFeature('CLOUD_IPPBX_OWN_SERVICE');
    }

    public static function hasPCOwnService()
    {
        return self::getInstance()->getBoolFeature('QX1500_IPPBX_OWN_SERVICE');
    }

    public static function hasVoicemailOwnService()
    {
        return self::getInstance()->getBoolFeature('VMS_OWN_SERVICE');
    }

    public static function hasIserv()
    {
        return self::getInstance()->getBoolFeature('ISERV');
    }

    public static function hasMediaStreamer()
    {
        return self::getInstance()->getBoolFeature('IPPBX_MEDIASTREAMER');
    }

    public static function isGateway()
    {
        return self::isPstnGateway() || self::isPhoneGateway();
    }

    public static function isPbx()
    {
        return self::getInstance()->match('PRODUCT_CLASS', 'PRODUCT_CLASS_IPPBX');
    }

    public static function isPc()
    {
        return !self::isSunxi();
    }

    public static function isPhoneGateway()
    {
        return self::getInstance()->match('PRODUCT_CLASS', 'PRODUCT_CLASS_PHONE_GATEWAY');
    }

    public static function isPstnGateway()
    {
        return self::getInstance()->match('PRODUCT_CLASS', 'PRODUCT_CLASS_PSTN_GATEWAY');
    }

    public static function isIsdnGateway()
    {
        return self::getInstance()->match('SUBPRODUCT_TYPE', 'QUADRO_A10_ISDNGW');
    }

    public static function isSunxi()
    {
        return self::getInstance()->match('HW_TYPE', 'HW_TYPE_SUNXI');
    }

    public static function isqx20()
    {
	return file_exists ( "/etc/fiad-conf/isqx20" );
    }
    
    public static function isEnabledWebRtcGateway()
    {
        return self::getInstance()->getBoolFeature('WEBRTC_GW');
    }

    public static function showFirmwareManualUpload()
    {
	if(self::isPC()){
	    // value in /tmp/ramsize is in KB
		$ramsize=$min_ramsize=1048576;
	    if (file_exists("/tmp/ramsize")){
		$f=file("/tmp/ramsize");
		foreach($f as $line)
		{
		    $ramsize=intval($line);
		    break;
	        }
	    }
	    return ($ramsize >= $min_ramsize);
        }
        return true;
    }

    public static function isConfigEmulator()
    {
        return self::getInstance()->match('PRODUCT_SUBCLASS', 'PRODUCT_SUBCLASS_CONFIG_EMULATOR');
    }

    /**
     * @param $key - configuration key
     * @param $value - value to test against
     *
     * @return bool - true if config key exists and is set to the passed value
     */
    protected function match($key, $value)
    {
        return $this->get($key) == $value;
    }

    public static function isHosted()
    {
	return file_exists ( "/mnt/factory/factorysafedir/ishosted" );
    }

    public static function isH3Board()
    {
        if( self::isSunxi() )
        {
		  if( self::getHwVersion() == 2)
		  {
			return true;
		  }
		}
        return false;
    }

}
