<?php
namespace Mvc;

use Exception;
use Mvc\View\Placeholder;

class View
{
    const FILE_EXTENSION = 'phtml';
    const HELPER_NAMESPACE = '\App\Form\Helper\\';
    const LAYOUT_FOLDER = 'layout';
    const VIEW_FOLDER = 'view';

    // full path to template file
    protected static $_data = array();
    private $_view_script;

    // stores all variables passed to view
    private $_view_name;

    public function __construct($view_name = '')
    {
        // assign view
        if ($view_name) {
            $this->setView($view_name);
        }
    }

    public function __call($method, $args)
    {
        /**
         * Look for a user-defined view helper as a class named <HELPER_NAMESPACE>\<method> class.
         */
        static $plugins = array();

        // create new helper instance
        if (!isset($plugins[$method])) {
            // path to controller class
            $class = self::HELPER_NAMESPACE . ucfirst($method);

            // controller class does not exist!
            if (!class_exists($class)) {
                throw new Exception("View Helper {$class} does not exist.");
            }

            // load and initialize helper class
            $plugins[$method] = $plugin = new $class();
            $plugin->setView($this);
        }

        // helper has callable method
        if (is_callable($plugins[$method])) {
            return call_user_func_array($plugins[$method], $args);
        }

        // return instance to our helper
        return $plugins[$method];
    }

    /**
     * @param $key - property name
     *
     * @return string - property value (or empty string if not set)
     */
    public function __get($key)
    {
        return isset(self::$_data[$key]) ? self::$_data[$key] : '';
    }

    /**
     * Manages storing the data passed to the view as properties
     *
     * @param string $key Property name
     * @param string $val Property value
     */
    public function __set($key, $val)
    {
        self::$_data[$key] = $val;
    }

    public function clear()
    {
        self::$_data = array();
    }

    /**
     * HELPER: escape text for HTML output.
     *
     * @param $text - text to escape
     *
     * @return string - escaped text
     */
    public function escape($text)
    {
        return htmlspecialchars($text, ENT_QUOTES);
    }

    /**
     * HELPER: escape text for HTML output.
     *
     * @param $text - text to escape
     *
     * @return string - escaped html
     */
    public function escapeHtml($text)
    {
        return htmlspecialchars($text, ENT_QUOTES);
    }

    public function mergeParams($params)
    {
        // bogus input
        if (!is_array($params)) {
            return;
        }

        // merge params into our data
        self::$_data = array_merge(self::$_data, $params);
    }

    /**
     * HELPER: wrapper for placeholder object instances by name.
     *
     * @param $name - name of the placeholder object to fetch
     *
     * @return mixed - placeholder object
     */
    public function placeholder($name)
    {
        return Placeholder::getInstance($name);
    }

    /**
     * Renders the template.  Properties set will be available inside view file as variables through $this->varname.
     * @return string Rendered template
     */
    public function render()
    {
        // start capturing PHP output
        ob_start();

        // include view script
        include $this->_view_script;

        // return buffered output
        return ob_get_clean();
    }

    public function setLayout($view_name)
    {
            $this->setView($view_name, self::LAYOUT_FOLDER);
    }

    public function setView($view_name, $view_folder = self::VIEW_FOLDER)
    {
        // save view name
        $this->_view_name = $view_name;

        // find view script
        $view_script = Application::findAssetFile($view_folder, $view_name, self::FILE_EXTENSION);

        // unable to find view script
        if (!$view_script) {
            $view_script = Application::findAssetFile(self::LAYOUT_FOLDER, 'not-found', self::FILE_EXTENSION);
        }

        // save view script
        $this->_view_script = $view_script;
    }

    public function viewFile($view_name, $view_folder = self::VIEW_FOLDER)
    {
        return Application::findAssetFile($view_folder, $view_name, self::FILE_EXTENSION);
    }
}
