<?php
namespace Epygi\Storage;

use Exception;

class Extension
{
    private $extension_id = null;
    private $extension_type = null;

    public function __construct($ext_id, $type = false)
    {
        // we don't know the type, try to figure it out
        if (!$type) {
            $type = $this->findExtensionTypeById($ext_id);
        }
        // save extension id and type
        $this->extension_id = $ext_id;
        $this->extension_type = $type;
    }

    public static function add()
    {
    }

    public function conferenceConfig()
    {
        $file_name = 'conference-configs/conf/' . $this->extension_id . '.conf';
        $conf = ConfigFile::getInstance($file_name);
        $section = $conf['General'];
        return $section;
    }

    /**
     * @return bool - true is extension is found in extensiondb, false otherwise
     */
    public function exists()
    {
        return (boolean) $this->extension_type;
    }

    private function findExtensionTypeById($ext_id)
    {
        if (\Mvc\Feature::hasVoicemailOwnService()) {
            // we need to check sqlitedb
            return \Epygi\Query\Extension::getType($ext_id);
        }

        // extension db and key we are looking for
        $config = ConfigFactory::extensionDb();

        foreach ($config as $section) {
            // we found the extension in this section if ext id is in values for this section
            if (in_array($ext_id, $section->getArray())) {
                // our type is the name of the section (ex: conference, extension, recordingbox, etc)
                return $section->getName();
            }
        }

        // did not find the extension in any section
        return null;
    }

    /**
     * A simple way to access the ConfigFile for each config file for the selected extension.  Gracefully
     * handles returning the default configuration!  So, be careful you do not update the DEFAULT when you
     * use this method.  Suggested to treat configs as read-only when fetched this way!
     *
     * @param $config_name
     *
     * @return ConfigFile
     * @throws \Exception
     */
    public function getConfig($config_name)
    {
        if ($this->extension_type === 'voicemailbox')
            return $this->getJsonConfig($config_name);

        switch ($config_name) {
            case 'codecs':
                return $this->getConfigOrDefault('codecsdb');

            case 'personal':
                return $this->getConfigOrDefault('personaldatadb');

            case 'sip':
                return $this->getConfigOrDefault('sipsettingsdb');

            case 'voicemail':
                return $this->getConfigOrDefault('voicemaildb');
            default:
                throw new Exception('Unknown Config File Type: ' . $config_name);
        }
    }

    /**
     * Returns a ConfigFile object which is either the user's config, or the default config for this extension
     * type.
     *
     * @param $config_file_name
     *
     * @return ConfigFile
     */
    private function getConfigOrDefault($config_file_name)
    {
        //defatt is for default attendant (not a typo)
        $defdir = ($this->extension_type == 'attendant') ? 'defatt' : 'default';

        // codecs config (or default codecs based on type)
        $ext_config = 'telephony/' . $this->extension_id . '/' . $config_file_name;
        $default_config = 'telephony/' . $defdir . '/' . $config_file_name;
        return ConfigFile::findConfigFile($ext_config, $default_config);
    }

    /**
     * Returns an assoc array which is the voicemailbox's config.
     *
     * @param $config_name table name
     *
     * @return array
     */
    private function getJsonConfig($config_name)
    {
        switch ($config_name) {
            case 'codecs':
                $table_name = 'codecsdb';
                break;
            case 'personal':
                $table_name = 'personaldatadb';
                break;
            case 'sip':
                $table_name = 'sipsettingsdb';
                break;
            case 'voicemail':
                $table_name = 'voicemaildb';
                break;
            default:
                throw new Exception('Unknown Config File Type: ' . $config_name);
        }

        $json_config = \Epygi\Query\Extension::getJsonData($this->extension_id, $table_name);
        return json_decode($json_config, true);
    }


    /**
     * Return the extension type.  If extension is not valid, type is null.
     * @return bool|null
     */
    public function getExtensionType()
    {
        return $this->extension_type;
    }

    /**
     * Fetches list of extensions by given extension types.
     *
     * @param $types - array of extension types
     *
     * @return array
     */
    public static function getExtensionsByTypes($types)
    {
        $config = ConfigFactory::extensionDb();
        $data = array();

        foreach ($types as $type) {
            // fetch extension IDs for extension type
            $section = $config[$type];
            $ids = array_values($section->getArray());

            // sort extension id values naturally
            natsort($ids);

            // save ids for this type
            $data[$type] = array_values($ids);
        }

        return $data;
    }
}