<?php
namespace Epygi\Storage;

use ArrayAccess;
use Iterator;
use Mvc;

class MultisectionConfigFile implements ArrayAccess, Iterator
{
    private static $instances = array();

    // absolute file path
    public $abs_file_path = null;

    // store all section objects here
    public $_root = null;
    private $_sections = array();

    /**
     * Create new object from a file name and default to parsing the config file data into a navigable object.
     *
     * @param null $file_name
     * @param bool $read
     */
    private function __construct($file_name, $read = true)
    {
        // we have a file name, load the configs if possible
        if ($file_name) {
            // convert relative file name to absolute path
            $this->abs_file_path = self::getConfigFilePath($file_name);

            // parse the config file
            if ($read) {
                // parse the config file
                $parser = new MultisectionConfigParser($this->abs_file_path);
                $sections = $parser->parse();

                // save root block and sections
                $this->_root = array_shift($sections);
                $this->_sections = $sections;
            }
        }

        // start empty
        if (!$this->_root) {
            $this->_root = new ConfigSection('ROOT');
        }

        // cache this instance for later
        self::$instances[$file_name] = $this;
    }

    /**
     * Convert an array to  config file and save it to filesystem
     *
     * @param $array
     *
     * @return array|string
     */
    public function saveArrayToFile($array)
    {
        foreach ($array as $v) {
            array_walk(
                $v,
                function (&$val, $k) {
                    $val = sprintf('%s=%s', $k, $val);
                }
            );
            $data[] = sprintf("{\n%s\n}", implode("\n", $v));
        }
        $data = implode("\n", $data);

        $text = $data . PHP_EOL; //"\r\n";
        return file_put_contents($this->abs_file_path, $text);
    }

    /**
     * ROOT BLOCK - pass through get to root block
     *
     * @param $key
     *
     * @return mixed|void
     */
    public function __get($key)
    {
        return $this->_root->$key;
    }

    /**
     * ROOT BLOCK - pass through key check to root block
     *
     * @param $key
     *
     * @return bool|void
     */
    public function __isset($key)
    {
        return isset($this->_root->$key);
    }

    /**
     * ROOT BLOCK - pass through set to root block
     *
     * @param $key
     * @param $value
     *
     * @return bool|float|int|string|void
     */
    public function __set($key, $value)
    {
        return $this->_root->$key = $value;
    }

    /**
     * convert the internal data structure back to a config file
     */
    public function __toString()
    {
        $out = array();

        // dump root object
        $data = $this->_root->getArray();
        //uksort($data, 'strnatcmp');

        foreach ($data as $key => $value) {
            $out[] = sprintf('%s %s', $key, $value);
        }

        // sections
        /*foreach ($this->_sections as $section) {
            // section start
            $out[] = trim(sprintf('%s %s {', $section->getLabel(), $section->getName()));

            // params
            $data = $section->getArray();
            uksort($data, 'strnatcmp');
            foreach ($data as $key => $value) {
                $out[] = sprintf('%s=%s', $key, $value);
            }

            // section end
            $out[] = '}';
        }*/
        $out = $this->sectionToArray($out, $this->_sections);

        // render
        return join(PHP_EOL, $out) . PHP_EOL;
    }

    private function sectionToArray($out, $sections)
    {
        foreach ($sections as $section) {
            // section start
            $out[] = trim(sprintf('%s %s {', $section->getLabel(), $section->getName()));

            // params
            $data = $section->getArray();
            //uksort($data, 'strnatcmp');
            foreach ($data as $key => $value) {
                $out[] = sprintf('%s %s', $key, $value);
            }

            $subs = $section->getSub();
            $out = $this->sectionToArray($out, $subs);

            // section end
            $out[] = '}';
        }
        return $out;
    }

    /**
     * ROOT BLOCK - pass through key removal to the root block
     *
     * @param $key
     */
    public function __unset($key)
    {
        unset($this->_root->$key);
    }

    /**
     * Iterator - current section position
     * @return mixed
     */
    public function current()
    {
        return current($this->_sections);
    }

    /**
     * Given a list of config file names, we return a ConfigFile object for the first config file which exists.
     * This function can be used to find settings for a specific extension or resort to loading the defaults if the
     * settings are not defined on the extension.
     * @return null|ConfigFile
     */
    public static function findConfigFile()
    {
        // the parameters to this function are a list of file names relative to the config directory
        $file_names = func_get_args();
        // loop through the list of file names
        foreach ($file_names as $file_name) {
            // calculate absolute path to the file
            $abs_file = self::getConfigFilePath($file_name);

            // the config file *does* exist, hooray, let's use this one
            if (file_exists($abs_file)) {
                return self::getInstance($file_name);
            }
        }

        // no config file found!
        return null;
    }

    /**
     * Loop through sections array and find the array index for the first section we encounter which has the section
     * name we are looking for.
     *
     * @param $section_name - section name
     *
     * @return null|int - position in our section array
     */
    private function findSectionByName($section_name)
    {
        // return the first section with a matching name
        foreach ($this->_sections as $index => $section) {
            if ($section->getName() == $section_name) {
                return $index;
            }
        }

        // no section with matching name
        return null;
    }

    public function get($key, $default = null)
    {
        return $this->_root->get($key, $default);
    }

    public function getBool($key, $default = false)
    {
        return $this->_root->getBool($key, $default);
    }

    /**
     * Convert the name of the config file to an absolute file path by automatically appending our config directory
     * name to the relative file name path.
     *
     * @param $file_name
     *
     * @return string
     */
    private static function getConfigFilePath($file_name)
    {
        // file name looks like it is absolute already
        if (substr($file_name, 0, 1) === '/') {
            return $file_name;
        }

        // convert relative file name to absolute path
        return ConfigFile::QUADRODB_ROOT . DIRECTORY_SEPARATOR . $file_name;
    }

    public static function getInstance($file_name)
    {
        // instance already created
        if (isset(self::$instances[$file_name])) {
            return self::$instances[$file_name];
        }

        // create new instance and cache it
        return new self($file_name);
    }

    public static function reLoad($file_name)
    {
        return new self($file_name);
    }

    /**
     * Iterator -
     */
    public function key()
    {
        return key($this->_sections);
    }

    /**
     * Iterator -
     * @return mixed
     */
    public function next()
    {
        // keep advancing until we are happy
        do {
            $section = next($this->_sections);

            // found non-empty section
            if ($section && !$section->isEmpty()) {
                return $section;
            }
            // while not end of array
        } while ($this->valid());

        // we reached the end of the array (no non-empty section found)
        return $section;
    }

    /**
     * ArrayAccess - does section exist?
     *
     * @param $offset
     *
     * @return bool
     */
    public function offsetExists($offset)
    {
        // search through the sections to see if we have any with a matching name
        foreach ($this->_sections as $section) {
            if ($section->getName() == $offset) {
                return true;
            }
        }

        // no matching names found
        return false;
    }

    /**
     * ArrayAccess - fetch a section block (side effect will create empty section)
     *
     * @param $offset
     *
     * @return null
     */
    public function offsetGet($offset)
    {
        $index = $this->findSectionByName($offset);

        // found it!
        if (!is_null($index)) {
            return $this->_sections[$index];
        }

        // no section with matching name found ... create a new section
        $section = new ConfigSection($offset);

        // add this section to the end of our section array
        return $this->_sections[] = $section;
    }

    /**
     * ArrayAccess - set a section
     *
     * @param mixed $offset - section name to replace
     * @param mixed $value - replacement value
     *
     * @throws Exception
     */
    public function offsetSet($offset, $value)
    {
        // value is an array ... convert it to ConfigSection
        if (is_array($value)) {
            $value = new ConfigSection($offset, null, $value);
        } // otherwise, we only accept config sections
        elseif (!($value instanceof ConfigSection)) {
            throw new Exception ('invalid config file section');
        }

        // find an existing section if we should replace one
        $index = $offset ? $this->findSectionByName($offset) : null;

        // nothing to replace ... just append it
        if (is_null($index)) {
            $this->_sections[] = $value;
            return;
        }

        // replace existing section block
        $this->_sections[$index] = $value;
    }

    /**
     * ArrayAccess = remove section
     *
     * @param $offset
     */
    public function offsetUnset($offset)
    {
        // find index of first matched section
        $index = $this->findSectionByName($offset);

        // remove offset from our sections
        if (!is_null($index)) {
            unset($this->_sections[$index]);
        }
    }

    /**
     * Iterator -
     * @return bool
     */
    public function rewind()
    {
        return reset($this->_sections);
    }

    /**
     * Root node, to make accessing it easier.
     * @return ConfigSection
     */
    public function root()
    {
        return $this->_root;
    }

    /**
     * Render the object as a string and write that string content to our original file.
     * @return int - status of write
     */
    public function save()
    {
        $text = (string) $this . PHP_EOL; //"\r\n";
        return file_put_contents($this->abs_file_path, $text);
    }

    /**
     * List of sections in case we need to iterate through them.
     * @return array
     */
    public function sections()
    {
        return $this->_sections;
    }

    /**
     * Iterator -
     * @return bool
     */
    public function valid()
    {
        $key = key($this->_sections);
        return ($key !== null && $key !== false);
    }

    /**
     * Create config file if not exist
     *
     * @param string $file_name
     *
     * @return bool return true if file exist of created normally, false if file not exist and can't to create file
     */

    public static function createConfigFile($file_name)
    {
        if (!file_exists($file_name)) {
            return touch($file_name);
        }
        return true;
    }
}