<?php
namespace Epygi\Shell;

use Epygi;
use Mvc;

class Passthru extends Command
{
    public function __construct($command)
    {
        // parent constructor (Command)
        parent::__construct($command);

        /**
         * this is important! We must turn off output buffering and flush all existing buffers. Then, the
         * output MUST be set to text/html for browsers to respond appropriately.
         */
        header('Content-type: text/html; charset=utf-8');
        ob_end_clean();
        ob_implicit_flush(true);
    }

    public function executeMessageQueue($queue_key)
    {
        // execute the command
        $result = $this->execute();

        // start html output
        $this->htmlBegin();

        // connect to message queue with given queue key (default permissions 0666)
        $queue = msg_get_queue($queue_key, 0666);

        // loop while receiving messages from queue
        $buffer = null;
        while (msg_receive($queue, 0, $msg_type, 1024, $message, false, 0, $err_code)) {
            // if we receive an error, stop immediately
            if ($err_code) {
                $this->htmlWrite('Received ' . $err_code . ' fetching message.', true);
                break;
            }
            // normal message (the 'd' character)

            if ($msg_type == 100) {
                // remove last 3 characters from end of message (why?)
		if( strlen($message) > 3) {
            	    $message = substr($message, 0, strlen($message) - 3);
		}

                if( strstr($message,'\\n') ) {
                    // split message on line breaks
                    $lines = preg_split('/\\\n/', $buffer . $message, 0);
                    // text after a line break stays in the buffer
                    $buffer = array_pop($lines);

                    // lines ready to send to browser (after a newline break)
                    foreach ($lines as $line) {
                        $this->htmlWrite($line, true);
                    }
                }
                else {
                    $this->htmlWrite($message, false);
                }
            }
            // close page message (the 'c' character)
            elseif ($msg_type == 99) {
                break;
            }
        }

        // send remaining buffer
        if ($buffer) {
            $this->htmlWrite($buffer, true);
        }

        // clean up message queue
        msg_remove_queue($queue);

        // end html output
        $this->htmlEnd();

        return $result;
    }

    /**
     * Open the command as a multi-ended pipe that we can read and write to ... then loop through the output of
     * the command and start sending directly to the browser (flush after each line).
     */
    public function executePassthru()
    {
        // render our command as a string
        $command = 'LANG=' . Epygi\Locale::getLocale() . ' ' . (string) $this;

        // TODO: handle errors from executed commands.  If proc_open fails ... what then?
        $process = proc_open(
            $command,
            array(
                0 => array('pipe', 'r'), // stdin is a pipe that the child will read from
                1 => array('pipe', 'w'), // stdout is a pipe that the child will write to
                2 => array('pipe', 'w') // stderr is a pipe that the child will write to
            ),
            $pipes,
            '/tmp',
            array()
        );

        // start html output
        $this->htmlBegin();

        // we have a process
        if (is_resource($process)) {
            while ($buffer = fgets($pipes[1])) {
                $this->htmlWrite($buffer);
            }
            fclose($pipes[0]);
            fclose($pipes[1]);
            proc_close($process);
        }

        // no process!
        else {
            $this->htmlWrite(_('error!'));
        }

        // end html output
        $this->htmlEnd();
    }

    /**
     * In another life, we will put this into a template file and not embed the style directly into this class, but
     * for today, we are feeling rushed and little naughty.  Git-er-done!
     */
    protected function htmlBegin()
    {
        echo '<html>';
        echo '<head>';
        echo '<style>div { font: normal 13px/18px Consolas, Lucida Console, monospace; white-space: pre; }</style>';
        echo '</head>';
        echo '<body>';
        echo "\n";
        echo "<div>";
        flush();
    }

    protected function htmlEnd()
    {
        // write finish statement
        $this->htmlWrite(_('Done.'));

        echo "</div>";
        // end html
        echo '</body>';
        echo '</html>';
        echo "\n";
        flush();
    }

    private function htmlWrite($message, $newline=false)
    {
        // replace tabs with spaces
        $message = preg_replace('/\t/', '   ', $message);
        print trim($message) ? htmlspecialchars($message) : '&nbsp;';
        if($newline) {
            print "\n";
        }
        flush();
    }
}
