<?php
namespace Epygi\Db\Statement;

class Helper
{
    // internal
    private $_map = array();
    private $_type = array();

    // filter
    private $_filters = array();

    // order
    private $_order_column = '';
    private $_order_dir = 'ASC';

    // pager
    private $_limit = 0;
    private $_offset = 0;

    public function __construct()
    {
        // initialize mappings and types
        $this->_map = array();
        $this->_type = array();
    }

    public function add($alias, $column = '', $type = '')
    {
        $alias = strtolower($alias);

        // save alias to column mapping
        $this->_map[$alias] = $column ? strtolower($column) : $alias;

        // save data type mapping
        $this->_type[$alias] = $type;

        // chaining
        return $this;
    }

    public function addFilter($field, $value, $type = '')
    {
        // unknown filter field!  abort!
        $field = strtolower($field);
        if (!isset ($this->_map[$field])) {
            return false;
        }

        // load type from definition if not explicitly passed in
        if (!$type) {
            $type = $this->_type[$field];
        }

        // convert field name
        $field = $this->_map[$field];

        // save filter
        $this->_filters[] = array(
            'field' => $field,
            'value' => $value,
            'type' => $type
        );

        // filter added
        return true;
    }

    public function getFilters()
    {
        return $this->_filters;
    }

    public function getLimit()
    {
        return $this->_limit;
    }

    public function getOffset()
    {
        return $this->_offset;
    }

    public function getOrderColumn()
    {
        return $this->_order_column;
    }

    public function getOrderDir()
    {
        return $this->_order_dir;
    }

    public function getSqlLimitOffset()
    {
        // read limit and offset from helper
        $limit = $this->getLimit();
        $offset = $this->getOffset();

        // do not add clause if no limit imposed
        if (!$limit) {
            return '';
        }

        // write limit/offset SQL line
        return sprintf('LIMIT %d OFFSET %d', $limit, $offset);
    }

    public function getSqlOrderBy()
    {
        // read limit and offset from helper
        $column = $this->getOrderColumn();
        $dir = $this->getOrderDir();

        // do not add clause if no limit imposed
        if (!$column) {
            return '';
        }

        // write order by SQL line
        return sprintf(' ORDER BY %s %s ', $column, $dir);
    }

    public function hasFilter()
    {
        return count($this->_filters) > 0;
    }

    /**
     * Set the limit and offset values for this query by passing the values in directly.  This does not take into
     * consideration the page number or page length.  You will need to use the "setPageLimitOffset" function to
     * convert from page number + page length to limit + offset.
     */
    public function setLimitOffset($limit, $offset, $min_limit = 0, $max_limit = 0)
    {
        // fix data types
        $limit = (int) sprintf('%d', $limit);
        $offset = (int) sprintf('%d', $offset);

        // limit can not be smaller than this min value
        if ($min_limit) {
            $limit = max($limit, $min_limit);
        }

        // limit can not be greater than this max value
        if ($max_limit) {
            $limit = min($limit, $max_limit);
        }

        // offset should never be negative
        $offset = max($offset, 0);

        // save values
        $this->_limit = $limit;
        $this->_offset = $offset;
    }

    public function setOrderBy($column, $direction = 'asc')
    {
        $column = strtolower($column);

        // unknown sort column!  abort!
        if (!isset ($this->_map[$column])) {
            return false;
        }

        // we want descending order
        if ($direction === -1 || strtoupper($direction) === 'DESC') {
            $direction = 'DESC';
        } // default to ascending otherwise
        else {
            $direction = 'ASC';
        }

        // save values
        $this->_order_column = $this->_map[$column];
        $this->_order_dir = $direction;
    }

    public function setPageLimitOffset($page_number, $page_length)
    {
        // convert page number and length to limit and offset
        $limit = $page_length;
        $offset = max($page_number - 1, 0) * $page_length;

        // set the limit and offset
        $this->setLimitOffset($limit, $offset);
    }
}