<?php
namespace Epygi\Grid;

use Mvc;

class Order
{
    const ORDER_FIELD = 'ocol';
    const ORDER_DIR = 'odir';

    // order field and sort direction
    private $field = '';
    private $direction = 0;

    /**
     * Initialize the order object by selecting a default field and default direction to sort on.
     *
     * @param $default_field - field name
     * @param $default_direction - direction
     */
    public function init($default_field, $default_direction)
    {
        // session storage
        $session = new Mvc\Session(Mvc\Router::getInstance()->getUri());

        // read and cache values from $_REQUEST or use defaults
        $this->field = $session->requestFilter(self::ORDER_FIELD, $default_field);
        $this->direction = $session->requestFilter(self::ORDER_DIR, $default_direction);
    }

    /**
     * Which field is being sorted.
     * @return string - field name
     */
    public function getField()
    {
        return $this->field;
    }

    /**
     * Return session-cached sort direction for the selected field.  We ensure our direction is 1 or -1.
     * @return int - sort direction 1 = ASC, -1 = DESC
     */
    public function getDirection()
    {
        return ($this->direction > 0) ? 1 : -1;
    }

    /**
     * Sort the input data on the given column and in the given direction based on a natural case-insensitive
     * comparison method.
     *
     * @param $input - array of data to sort
     */
    public function order(&$input)
    {
        // direction multiplied by case comparison result
        $direction = $this->getDirection();

        // user-defined custom sort for input array
        usort(
            $input,
            function ($a, $b) use ($direction) {
                // sort field missing, no check
                if (!isset($a[$this->field]) || !isset($b[$this->field])) {
                    return 0;
                }

                // sort fields exist, compare values
                return $direction * strnatcasecmp($a[$this->field], $b[$this->field]);
            }
        );
    }
}