<?php
namespace Epygi;

use Mvc;

class Locale
{
    const LOCALE_COOKIE_NAME = 'epyloc';
    const LOCALE_DIR = '/usr/share/locale';

    private static $locale = 'en_US';

    private static function existsLocale($locale, $translation_file_only = false)
    {
        // this locale is the hard-coded default (it always exists)
        if (!$translation_file_only) {
            if ($locale == 'en_US' || $locale == 'C') {
                return true;
            }
        }

        // translations file for this locale
        $translation_file = self::LOCALE_DIR . '/' . $locale . '/LC_MESSAGES/epygi.mo';

        // does translation file exist?
        return file_exists($translation_file);
    }

    public static function findAvailableLocales()
    {
        // open the locale config and loop through all sections
        $cfg = Storage\ConfigFactory::locale();

        $available_locales = array();
        foreach ($cfg as $locale) {
            $name = $locale->getName();

            // if the locale exists or is the default "C" locale, add to our list
            if ($name === 'C' || self::existsLocale($name)) {
                $available_locales[$name] = array(
                    'locale' => $name == 'C' ? 'en_US' : $name,
                    'name' => $locale->get('name'),
                    'charset' => $locale->get('charset')
                );
            }
        }

        // these are our valid locales (probably only 1 or 2 items in the list)
        return $available_locales;
    }

    public static function getLocale()
    {
        return self::$locale;
    }

    public static function getLocaleName($code)
    {
        $codes = array(
            'C' => 'English',
            'de_DE' => 'Deutsch',
            'el_GR' => 'Ελληνικά',
            'en_AU' => 'English',
            'en_US' => 'English',
            'es_AR' => 'Español',
            'es_ES' => 'Español',
            'es_INTL' => 'Español',
            'es_MX' => 'Español',
            'es_VE' => 'Español',
            'fr_FR' => 'Français',
            'hy_AM' => 'Հայերեն',
            'it_IT' => 'Italiano',
            'ja_JP' => '日本',
            'lt_LT' => 'Lietuvos',
            'pl_PL' => 'Polski',
            'pt_BR' => 'Português',
            'pt_DA' => 'Português',
            'pt_PT' => 'Português',
            'ro_RO' => 'Român',
            'ru_RU' => 'Русский',
            'sl_SI' => 'Slovenščina',
	    'he_IL' => 'עברית',
        );

        return isset($codes[$code]) ? $codes[$code] : 'English';
    }

    /**
     * On every page load, the bootstrap.php file will call this function to set the locale for the page request.
     * We can use this opportunity to set a user's session locale or use the system default language.
     * @return string - selected locale
     */
    public static function selectLocale()
    {
        /**
         * if the user has set a new locale and that locale actually exists, we save the setting into both
         * the PHP session and CGI session so it can be used on every page request.
         */
        // if user has set a new locale and that locale actually exists, save it
        $user_locale = Mvc\Param::request('locale', null);
        if (!is_null($user_locale) && self::existsLocale($user_locale)) {
            // write value to PHP session
            self::storageCookieWrite($user_locale);

            // write value to CGI session
            $cgi_session_id = Auth::getCgiSessionId();
            $user_locale = ($user_locale == 'en_US') ? 'C' : $user_locale; // write 'C' instead of 'en_US'
            Storage\SessionData::setLocale($cgi_session_id, $user_locale);
        }

        // try to load the locale from user PHP session
        $locale_lang = isset ($_COOKIE[self::LOCALE_COOKIE_NAME]) ? $_COOKIE[self::LOCALE_COOKIE_NAME] : false;

        // if not found: load it from CGI session
        if(!$locale_lang) {
            $cgi_session_id = Auth::getCgiSessionId();
            $locale_lang = Storage\SessionData::getLocale($cgi_session_id);
            if(is_null($locale_lang) || empty($locale_lang)) {
                $locale_lang = false;
            }
        }
        // use session does not have a locale, use the system default
        if (!$locale_lang) {
            // which language is the system configured to use?
            $rc = Storage\ConfigFactory::rc();
            $locale_lang = $rc->get('LANG');
        }

        // set this locale
        return self::setLocale($locale_lang);
    }

    private static function setLocale($locale)
    {
        // if the translations file does not exist, do not bind translations
        if (!self::existsLocale($locale, true)) {
            return self::$locale;
        }

        // look up encoding in /usr/share/locale/locale.conf
        $cfg = Storage\ConfigFactory::locale();
        $section = $cfg[$locale];
        $charset = $section->get('charset');

        // can't determine the character set, abort!
        if (!$charset) {
            return self::$locale;
        }

        // set locale
        setlocale(LC_ALL, $locale . '.' . $charset); // . '.UTF-8');
        putenv('LC_ALL=' . $locale);
        putenv('LANGUAGE=' . $locale);

        // specify location of translation tables and bind domain
        bindtextdomain('epygi', self::LOCALE_DIR);
        textdomain('epygi');

        // return set locale
        return self::$locale = $locale;
    }

    private static function storageCookieWrite($locale)
    {
        // domain
        $regs = false;
        preg_match('/\.([^\.]+\.[^\.]+)$/', Mvc\Param::server('HTTP_HOST'), $regs);
        $domain = isset($regs[1]) ? $regs[1] : '';

        // set the cookie as HTTP header
        $expire = time() + (60 * 60 * 24 * 90); // expiration (90 days)
        setcookie(self::LOCALE_COOKIE_NAME, $locale, $expire, '/', $domain, true, true);

        // update cookie array
        $_COOKIE[self::LOCALE_COOKIE_NAME] = $locale;
    }

    public static function isLanguagePackInstalled()
    {
        return file_exists('/mnt/language/locale.conf');
    }
}