<?php
namespace Epygi\Grid;

use Mvc;

class Filter
{
    const FILTER_FIELD = 'fcol';
    const FILTER_VALUE = 'fval';
    const FILTER_CLEAR = 'fclear';

    // allowed filters
    private $filters = array();

    // submitted filter field
    private $filter_field = '';
    private $filter_value = '';

    /**
     * Defines the column "aliases" that can be used for searching.  Field will be the value passed as a param
     * 'filter-field' and label will be displayed in the UI drop-down.
     */
    public function add($field, $label)
    {
        $this->filters[$field] = $label;

        // chaining
        return $this;
    }

    public function getField()
    {
        return $this->filter_field;
    }

    public function getFilters()
    {
        return $this->filters;
    }

    public function getValue()
    {
        return $this->filter_value;
    }

    public function hasValue()
    {
        return strlen($this->getValue()) >= 1;
    }

    public function init()
    {
        // session storage
        $session = new Mvc\Session(Mvc\Router::getInstance()->getUri());

        // inport _REQUEST values and save in session cache
        $this->filter_field = $session->requestFilter(self::FILTER_FIELD);
        $this->filter_value = $session->requestFilter(self::FILTER_VALUE);
    }

    /**
     * Narrow down the input data by filtering a specific column by a specific value.  If the search field is "all",
     * all fields are searched and filtered.  The fastest search we can imagine uses 'stripos' in PHP.
     *
     * @param $input - input data
     *
     * @return array - output data
     */
    public function filter($input)
    {
        // nothing to do
        if (!$this->hasValue()) {
            return $input;
        }

        return array_filter(
            $input,
            function ($a) {
                // perform search through "all" columns
                if ($this->filter_field == 'all') {
                    foreach ($a as $value) {
                        if (stripos($value, $this->filter_value) !== false) {
                            return true;
                        }
                    }
                }

                // no such field to search on
                if (!isset($a[$this->filter_field])) {
                    return false;
                }

                // search term was found in the search field
                if (stripos($a[$this->filter_field], $this->filter_value) !== false) {
                    return true;
                }

                return false;
            }
        );
    }
}
