<?php
namespace Epygi\Grid;

use Mvc;

class Pager
{
    const DEFAULT_PAGE_LENGTH = 50;
    const PARAM_PAGE_NUMBER = 'pgnum';
    const PARAM_PAGE_LENGTH = 'pglen';

    // internals
    private $page_number = 1;
    private $page_length = 2;
    private $page_max = 1;
    private $record_count = 0;

    /**
     * Configures the Pager object based on a given record count.  Selected page number and page length values come
     * from $_REQUEST, $_SESSION, and DEFAULTS in that order.  Values are saved back to the session for future page
     * loads.
     *
     * @param $record_count
     */
    public function init($record_count)
    {
        // session storage
        $session = new Mvc\Session(Mvc\Router::getInstance()->getUri());

        // requested page number
        $this->page_number = $session->requestFilter(self::PARAM_PAGE_NUMBER, 1);

        // requested page length (make sure its an allowed value, or use default)
        $this->page_length = $session->requestFilter(self::PARAM_PAGE_LENGTH, self::DEFAULT_PAGE_LENGTH);
        if (!in_array($this->page_length, $this->optionPageLength())) {
            $this->page_length = self::DEFAULT_PAGE_LENGTH;
            $session->set(self::PARAM_PAGE_LENGTH, $this->page_length);
        }

        // calculate max page
        $this->page_max = ceil($record_count / $this->page_length);

        // make sure requested page falls within bounds
        $this->page_number = min(max($this->page_number, 1), $this->page_max);

        // save page number again (it may have changed)
        $session->set(self::PARAM_PAGE_NUMBER, $this->page_number);

        // save record count in pager because it may change in grid engine later after pager kicks in
        $this->record_count = $record_count;
    }

    /**
     * Get number of records to display on each page.
     * @return int - page length
     */
    public function getPageLength()
    {
        return $this->page_length;
    }

    /**
     * Get number of pages (last page number).
     * @return int - total number of pages
     */
    public function getPageMax()
    {
        return $this->page_max;
    }

    /**
     * This is a stupid function.  It's included to make the obvious point.
     */
    public function getPageMin()
    {
        return 1;
    }

    /**
     * Get the current selected page number.
     * @return int - page number
     */
    public function getPageNumber()
    {
        return $this->page_number;
    }

    /**
     * Get original number of records present before pager limits invoked.
     * @return int - record count
     */
    public function getRecordCount()
    {
        return $this->record_count;
    }

    /**
     * Slice the input array to select just the items which fit on the selected page.
     *
     * @param $input - input data
     *
     * @return array - output data
     */
    public function limitOffset($input)
    {
        return array_slice($input, ($this->page_number - 1) * $this->page_length, $this->page_length);
    }

    /**
     * Get list of all possible page lengths offered in the GUI.
     * @return array - page length options
     */
    public function optionPageLength()
    {
        return array(10, 15, 20, 25, 50, 100, 200);
    }
}