<?php
namespace Epygi\Storage;

use Countable;
use Exception;

class ConfigSection implements Countable
{
    // section label
    private $_label = null;

    // section name
    private $_name = null;

    // key=value configuration data
    private $_data = array();

    // subsection
    private $_sub = array();

    public function __construct($name = null, $label = null, $data = null)
    {
        // save block name
        if ($name) {
            $this->_name = $name;
        }

        // save block label
        if ($label) {
            $this->_label = $label;
        }

        // set data values
        if ($data) {
            // set data from an array
            if (is_array($data)) {
                $this->_data = $data;
            }
            // set data by cloning data from an object like this
            elseif ($data instanceof self) {
                $this->_data = $data->getArray();
                $this->_name = $data->getName();
                $this->_label = $data->getLabel();
            }
        }
    }

    /**
     * Object - get value
     *
     * @param $name
     *
     * @return mixed
     */
    public function __get($name)
    {
        return isset($this->_data[$name]) ? $this->_data[$name] : null;
    }

    /**
     * Object - check for existence of key
     *
     * @param $name
     *
     * @return bool
     */
    public function __isset($name)
    {
        return isset($this->_data[$name]);
    }

    /**
     * Object - set value
     *
     * @param $name
     * @param $value
     *
     * @return bool|float|int|string
     * @throws Exception
     */
    public function __set($name, $value)
    {

        // set nothing if empty key
        $name = trim($name);
        if (!$name) {
            return false;
        }

        // sanity check for data type
        if (!is_scalar($value)) {
            throw new Exception('Only store scalar values in section blocks!');
        }

        // save data type
        return $this->_data[$name] = $value;
    }

    /**
     * Object - remove key
     *
     * @param $name
     */
    public function __unset($name)
    {
        unset($this->_data[$name]);
    }

    public function count()
    {
        return count($this->_data);
    }

    /**
     * Function wrapper around getter in case key name is not proper PHP variable (might have spaces) or we want to
     * use a default value when the key is not found.
     *
     * @param $key
     * @param null $default
     *
     * @return null
     */
    public function get($key, $default = null)
    {
        return isset($this->_data[$key]) ? $this->_data[$key] : $default;
    }

    /**
     * Fetch all key/value pairs as an array (actually this is our internal data structure,
     * anyhow ... so it is very fast).
     * @return array - entire key/value pair array
     */
    public function getArray()
    {
        return $this->_data;
    }

    /**
     * Return the config value as a boolean.
     */
    public function getBool($key, $default = false)
    {
        $v = strtolower($this->get($key, $default));

        // we can only test scalar values
        if (!is_scalar($v)) {
            return (boolean) $v;
        }

        // clearly it is TRUE
        if ($v == 'yes' || $v == 'true' || $v == 'y' || $v == 'on' || $v === true) {
            return true;
        }

        // clearly it is FALSE
        if ($v == 'no' || $v == 'false' || $v == 'n' || $v == 'off' || $v === false) {
            return false;
        }

        // its not clear, just cast it and be done with it
        return (boolean) $v;
    }

    public function getInt($key, $default = false)
    {
        return (int) $this->get($key, $default);
    }

    /**
     * @return string - name of section (empty if no label)
     */
    public function getLabel()
    {
        // TODO: is this really true?
        // if we have a section name, we need a section label.  If we need a section label and we don't have one,
        // we'll use 'section' hard-coded
        return $this->_label; // $this->_name ? ($this->_label ? : 'section') : $this->_label;
    }

    /**
     * @return string - name of the block
     */
    public function getName()
    {
        return $this->_name;
    }

    /**
     *
     */
    public function getSub()
    {
        return $this->_sub;
    }

    /**
     * Does this block have any key/value pairs defined?
     * @return bool
     */
    public function isEmpty()
    {
        return count($this->_data) < 1;
    }

    /**
     * Function wrapper around setter in case key name is not proper PHP variable (might have spaces).
     *
     * @param $key
     * @param $value
     */
    public function set($key, $value)
    {
        $this->$key = $value;
    }

    /**
     * Sets multiple values from an array of key/value pairs.
     *
     * @param $data - array of key/value pairs
     */
    public function setAll($data)
    {
        foreach ($data as $key => $value) {
            $this->set($key, $value);
        }
    }

    /**
     * Sets the section label to something other than 'section'
     *
     * @param $label
     */
    public function setLabel($label)
    {
        $this->_label = $label;
    }

    /**
     *
     */
    public function setSub($subsection)
    {
        $this->_sub[] = $subsection;
    }
}