<?php
namespace Epygi;

use Mvc;

class Url
{
    // internal URL parts
    private $scheme = '';
    private $user = '';
    private $pass = '';
    private $host = '';
    private $port = '';
    private $path = '';
    private $query = '';
    private $fragment = '';

    // query string args
    private $ARGS = array();

    public function __construct($url = '', $args = false)
    {
        // separate function arguments
        $args = func_get_args();
        // nothing to do
        if (!count($args)) {
            return;
        }

        // parse the URL
        $url = (string) array_shift($args);
        $parts = parse_url($url);

        // import parts into this object
        if (is_array($parts)) {
            foreach ($parts as $key => $value) {
                $this->$key = $value;
            }
        }

        // break up the query string
        if ($this->query) {
            parse_str($this->query, $this->ARGS);
            $this->query = null;
        }

        // add query string parameters
        if (count($args)) {
            call_user_func_array(
                array(
                    $this,
                    'append'
                ),
                $args
            );
        }
    }

    /**
     * Fetch a query string argument from the URL.
     */
    public function __get($key)
    {
        return isset ($this->ARGS[$key]) ? $this->ARGS[$key] : false;
    }

    /**
     * Set a query string argument into the URL
     */
    public function __set($key, $value)
    {
        $this->ARGS[$key] = $value;
    }

    public function __toString()
    {
        // scheme
        $uri = $this->scheme ? $this->scheme . ':' . (($this->scheme == 'mailto') ? '' : '//') : '';

        // user & password
        $uri .= $this->user ? $this->user . ($this->pass ? ':' . $this->pass : '') . '@' : '';

        // host
        $uri .= $this->host ? $this->host : '';

        // port
        $uri .= $this->port ? ':' . $this->port : '';

        // path
        $uri .= $this->path ? $this->path : '';

        // query
        $uri .= (count($this->ARGS) > 0) ? '?' . self::queryStr($this->ARGS) : '';

        // fragment
        $uri .= $this->fragment ? '#' . $this->fragment : '';
        return (string) $uri;
    }

    /**
     * Remove a query string parameter.
     */
    public function __unset($key)
    {
        unset ($this->ARGS[$key]);
    }

    public function append()
    {
        $params = array();
        $args = func_get_args();

        // add each of the attributes in key/value pairs
        while (count($args)) {
            $key = array_shift($args);

            // argument is already an array, treat as key=value array
            if (is_array($key)) {
                foreach ($key as $key2 => $value2) {
                    $this->ARGS[(string) $key2] = (string) $value2;
                }
                continue;
            }

            // attribute is a scalar key, next params *must* be the scalar value
            $value = array_shift($args);
            $this->ARGS[(string) $key] = (string) $value;
        }

        // done
        return $this;
    }

    public function attr()
    {
        $args = func_get_args();

        // if odd number of args, last arg is boolean flag
        if (count($args) % 2) {
            // flag is false, do nothing and quit
            if (!(boolean) array_pop($args)) {
                return;
            }
        }

        // add each of the query string attributes in key/value pairs
        for ($i = 0; $i < count($args); $i += 2) {
            if ($args[$i]) {
                $this->ARGS[$args[$i]] = $args[$i + 1];
            }
        }
    }

    public static function getCurrentUrl()
    {
        return new self(self::getPath() . '?' . Mvc\Param::server('QUERY_STRING'));
    }

    /**
     * Sometimes we use PHP_SELF and other times we have to use REQUEST_URI or some REWRITE_URI indexes.  Rather than
     * figure out which is which and have to rewrite all the code, we put the reference into a single function and we
     * can change it here later.
     */
    public static function getPath()
    {
        // SCRIPT_URL
        if (!empty ($_SERVER['SCRIPT_URL'])) {
            return $_SERVER['SCRIPT_URL'];
        }

        // REDIRECT_URL
        elseif (!empty ($_SERVER['REDIRECT_URL'])) {
            return $_SERVER['REDIRECT_URL'];
        }

        // REQUEST_URI
        elseif (!empty ($_SERVER['REQUEST_URI'])) {
            $p = parse_url($_SERVER['REQUEST_URI']);
            return $p['path'];
        }

        // broken!
        error_log(__FILE__ . " / " . __FUNCTION__ . ': Could not determine $_SERVER["SCRIPT_URL"].');
        die('broken!');
    }

    /**
     * Determines if the current uri path starts with the same string as the uri path provided.  I prefer to use
     * PHP_SELF, but it doesn't have the correct value for some reason.  Using REQUEST_URI instead.
     */
    public static function on()
    {
        // current page
        $current_path = self::getPath();

        $args = func_get_args();
        foreach ($args as $uri) {
            // clean uri
            $uri = self::removeQueryString($uri);

            // does the current page start with the URI passed in?
            $found = ($uri && $current_path) ? (strpos($current_path, $uri) === 0) : false;
            if ($found) {
                return true;
            }
        }

        // no match
        return false;
    }

    /**
     * Posted to http://php.net/manual/en/function.parse-str.php on July 6, 2003 by none other than myself :-)  It's
     * in the public domain, so bite me.
     */
    public static function queryStr($args)
    {
        $params = array();

        // join key/value pairs
        foreach ($args as $key => $value) {
            $params[] = $key . '=' . rawurlencode($value);
        }

        // if we want xhtml, change the join character when combining
        return join('&', $params);
    }

    /**
     * Takes off everything after the '?' character from the provided URI.
     */
    private static function removeQueryString($uri)
    {
        $pos = strpos($uri, '?');
        if ($pos > 0) {
            return substr($uri, 0, $pos);
        }
        return $uri;
    }

    public function toString()
    {
        return (string) $this;
    }
}
