<?php
namespace Mvc;

use ErrorException;
use Mvc\Exception\PageNotFoundException;

class Debug
{
    // array of logged items
    public $logged = array();

    // flag that determines if the errors are displayed
    public $display_errors = true;

    public static function debugIncludedFiles()
    {
        $included_files = get_included_files();
        foreach ($included_files as $index => $file_name) {
            self::dump($index . ' ' . $file_name);
        }
    }

    public static function dump($data)
    {
        // choose dump format (html or cli)
        $is_cli = defined('STDIN') && isset ($_SERVER['argc']);

        // HTML mode
        if (!$is_cli) {
            print '<pre>' . htmlspecialchars(print_r($data, true)) . '</pre>';
            return;
        }

        // CLI mode
        print_r($data);
        print "\n";
    }

    public static function hex_dump($data, $newline = "\n")
    {
        static $from = '';
        static $to = '';

        static $width = 16; # number of bytes per line

        static $pad = '.'; # padding for non-visible characters

        if ($from === '') {
            for ($i = 0; $i <= 0xFF; $i++) {
                $from .= chr($i);
                $to .= ($i >= 0x20 && $i <= 0x7E) ? chr($i) : $pad;
            }
        }

        $hex = str_split(bin2hex($data), $width * 2);
        $chars = str_split(strtr($data, $from, $to), $width);

        $offset = 0;
        print '<pre>';
        foreach ($hex as $i => $line) {
            $line = sprintf('%6X : %s [%s]%s', $offset, join(' ', str_split($line, 2)), $chars[$i], $newline);
            echo htmlspecialchars($line);
            $offset += $width;
        }
        print '</pre>';
    }

    /**
     * Initializes the error handler to convert PHP errors to Exceptions.
     */
    public function init()
    {
        set_error_handler(
            function ($error_number, $error_message, $error_file, $error_line) {
                throw new ErrorException($error_message, $error_number, 0, $error_file, $error_line);
            },
            E_ALL
        );
    }

    /**
     * Adds an item to the log.
     *
     * @param mixed $val Item to be logged
     *
     * @return void
     */
    public function log($val)
    {
        $this->logged[] = $val;
    }

    /**
     * Displays the error page. If you set $display_errors to false only a small error message will be displayed.
     *
     * @param \Exception $exception Exception to display
     *
     * @return void
     */
    public function renderExceptionPage($exception)
    {
        // turn off output buffering (if on)
        while (ob_get_level()) {
            ob_end_clean();
        };

        // major error
        $status = '503 Service Temporarily Unavailable';

        // TODO: only developers should see the 'debug' layout, production should use error-503 page
        $template = 'debug';

        // page not found
        if ($exception instanceof PageNotFoundException) {
            $status = '404 Not Found';
            $template = 'error-404';
        }

        // send HTTP error status
        header($_SERVER["SERVER_PROTOCOL"] . ' ' . $status);
        header("Status: {$status}");

        // silent errors!
        if (!$this->display_errors) {
            echo $status;
            return;
        }

        // use debugging view and render
        $view = new View();
        $view->setLayout($template);
        $view->clear();
        $view->exception = $exception;
        $view->log = $this->logged;
        echo $view->render();
        exit;
    }

    public static function sep($label = null)
    {
        self::dump(str_repeat('-', 80));
        if ($label) {
            self::dump($label . ' - ' . time());
        }
    }
}
