<?php
namespace Mvc;

use Epygi;
use ErrorException;
use Exception;
use Mvc\Exception\PageNotFoundException;

class Application
{
    const ASSETS_DIR = '/assets';
    const CONTROLLER_NAMESPACE = '\App\Controller\\';

    // instance classes
    protected $_param = array();

    /**
     * Finds path to specified file in the /assets folder
     *
     * @param $folder - folder to search (classes or views, etc)
     * @param $name - name of the file without extension
     * @param string $extension - file extension
     *
     * @return bool|string - full path to the file (false if not found)
     */
    public static function findAssetFile($folder, $name, $extension = 'phtml')
    {
        // abs file path
        $file = ROOT_DIR . self::ASSETS_DIR . '/' . $folder . '/' . $name . '.' . $extension;

        // check if file exists
        return file_exists($file) ? $file : false;
    }

    /**
     * Retrieve a URL parameter.
     */
    public function param($key = null, $default = null)
    {
        // all params
        if ($key == null) {
            return $this->_param;
        }

        // find value, or use default
        return isset($this->_param[$key]) ? $this->_param[$key] : $default;
    }

    /**
     * Displays the error page. If you set $display_errors to false only a small error message will be displayed.
     *
     * @param \Exception $exception Exception to display
     *
     * @return void
     */
    public function renderExceptionPage($exception)
    {
        // turn off output buffering (if on)
        while (ob_get_level()) {
            ob_end_clean();
        };

        // major error
        $status = '503 Service Temporarily Unavailable';

        // TODO: only developers should see the 'debug' layout, production should use error-503 page
        $template = 'debug';

        // page not found
        if ($exception instanceof PageNotFoundException) {
            $status = '404 Not Found';
            $template = 'not-found';
        }

        // send HTTP error status
        header($_SERVER["SERVER_PROTOCOL"] . ' ' . $status);
        header("Status: {$status}");

        // use debugging view and render
        $view = new View();
        $view->setLayout($template);
        $view->clear();
        $view->exception = $exception;
        echo $view->render();
        exit;
    }

    public function run()
    {
        // convert errors to exceptions
        set_error_handler(
            function ($error_number, $error_message, $error_file, $error_line) {
                throw new ErrorException($error_message, $error_number, 0, $error_file, $error_line);
            },
            E_ALL
        );

        // process HTTP request and send back response
        try {
            // which path did we request?
            $path = Param::server('REQUEST_URI');

            // match a route based on requested path and method
            $router = Router::getInstance();
            $this->_param = $router->routeMatch($path);

            //------------------------------
            // ACL: begin ACL permission checks for matches route
            $route_acl = $router->getAttributeRecursive('acl');

            // route requires ACL role and user is not logged in ... require login
            if ($route_acl && !Epygi\Auth::isAuth()) {
                Epygi\Auth::rejectUnlessAuth(false);
            }

            // route requires ACL role and user is not the proper role, pretend page doesn't exist
            if ($route_acl && !($route_acl & Epygi\Auth::getRole())) {
                throw new PageNotFoundException("Invalid Permissions!");
            }
            // ACL: end ACL permission checks
            //------------------------------

            // name of controller class to load
            $controller_name = $this->param('controller');
            $controller_class = self::CONTROLLER_NAMESPACE . ucfirst($controller_name) . 'Controller';

            // controller class does not exist!
            if (!class_exists($controller_class)) {
                throw new PageNotFoundException("Class {$controller_class} doesn't exist");
            }

            // load controller class
            $controller = new $controller_class($this);
	    if( null != $this->param('template')){
		$controller->setTemplate($this->param('template'));
	    }

            // run action and return response
            $output = $controller->run();

            // start output
            header('Content-Type: text/html; charset=utf-8');
            echo $output;
        }
        catch (Exception $e) {
            self::renderExceptionPage($e);
        }
    }
}