<?php
// This class uses vTiger REST API for third party integration
// see https://community.vtiger.com/help/vtigercrm/developers/third-party-app-integration.html

class VtigerModel_v1 extends BaseModel {

    const API_URL_PATH = "webservice.php";
    const KEY_VTIGER_CALL_FAIL = "vTigerCallFail";
    const KEY_SUCCESS = "success";
    const KEY_ERROR = "error";
    const KEY_RESULT = "result";
    const KEY_TOKEN = "token";
    const KEY_SESSION_NAME = "sessionName";
    const KEY_ID = "id";
    const KEY_CODE = "code";
    const KEY_MESSAGE = "message";
    const KEY_FIELDS = "fields";
    const KEY_CREATEABLE = "createable";
    const KEY_DELETEABLE = "deleteable";
    const KEY_UPDATEABLE = "updateable";

    const ASSIGNED_TO_ID_NOT_FOUND_ERR_CODE = "ID_NOT_FOUND_FOR_ASSIGNED_TO";
    private $ASSIGNED_TO_ID_NOT_FOUND_ERR_MSG = "AssignedTo ID for %s '%s' not found";

    const ARGUMENT_MAY_NOT_BE_EMPTY_ERR_CODE = "ARGUMENT_MAY_NOT_BE_EMPTY";
    private $ARGUMENT_MAY_NOT_BE_EMPTY_ERR_MSG = "%s argument may not be empty";

    private $token;             // Challenge token to be used for login
    private $session_id;        // Unique Identifier for the session
    private $assigned_to_id;    // vTiger CRM Assigned To ID

    public function __construct(ApiRequest $apiRequestObj) {
        parent::__construct($apiRequestObj);
        $this->token = null;
        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];
        $this->assigned_to_id = null;
    }

   /* GetToken endpoint
    * @request POST api/v1/Vtiger/GetToken
    * @parameters
    * Name              | Placement | Description
    * ------------------|-----------|------------------------------------
    * vTigerUrl         | Body      | Client's vTiger CRM URL
    * vTigerUserName    | Body      | Login User name for client's vTiger CRM
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "vTigerUserName": "armen@epygi.com",
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "token": "5e737511092dc",
    *       "serverTime": 1584624913,
    *       "expireTime": 1584625213
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetToken(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $vTigerUserName = $args['vTigerUserName'];

        $result = $this->challenge($apiUrl, $vTigerUserName);

        return array(ApiConstants::KEY_RAW_RESPONSE => $result);
    }

   /* GetSessionID endpoint
    * @request POST api/v1/Vtiger/GetSessionID
    * @parameters
    * Name                | Placement | Description
    * --------------------|-----------|-----------------------------------------------------------------------
    * vTigerUrl           | Body      | Client's vTiger CRM URL
    * vTigerUserName      | Body      | Login User name for client's vTiger CRM
    * vTigerUserAccessKey | Body      | Access Key (Can be found under “My Preferences” in the CRM Web UI.)
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "vTigerUserName": "armen@epygi.com",
    *   "vTigerUserAccessKey": "5Csw56e2Xh8O89Bx"
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "sessionName": "2709e6f5e7e11212333e",
    *       "userId": "19x6",
    *       "version": "0.22",
    *       "vtigerVersion": "7.2.0"
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetSessionID(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $vTigerUserName = $args['vTigerUserName'];
        $vTigerUserAccessKey = $args['vTigerUserAccessKey'];

        $result = $this->getSessionIDinternal($apiUrl, $vTigerUserName, $vTigerUserAccessKey);

        return array(ApiConstants::KEY_RAW_RESPONSE => $result);
    }

   /* ListTypes endpoint
    * @request POST api/v1/Vtiger/ListTypes
    * @parameters
    * Name                | Placement       | Description
    * --------------------|-----------------|-------------------------------------------
    * vTigerUrl           | Body            | Client's vTiger CRM URL
    * Authorization       | Headers         | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "types": [
    *         "Contacts",
    *         "ModComments",
    *         "Groups"
    *      ],
    *      "information": {
    *         "Contacts": {
    *           "isEntity": true,
    *           "label": "Contacts",
    *           "singular": "Contact"
    *         },
    *         "ModComments": {
    *           "isEntity": true,
    *           "label": "Comments",
    *           "singular": "Comment"
    *         },
    *         "Groups": {
    *           "isEntity": false,
    *           "label": "Groups",
    *           "singular": "Groups"
    *         }
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function ListTypes(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;

        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];

        $api = new RestAccess();
        $result = $api->get($apiUrl, array("operation" => "listtypes", "sessionName" => $this->session_id));
        unset($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        return array(ApiConstants::KEY_RAW_RESPONSE => $response);
    }

   /* Describe endpoint
    * @request POST api/v1/Vtiger/Describe
    * @parameters
    * Name           | Placement       | Description
    * ---------------|-----------------|----------------------------------------------------------------------
    * vTigerUrl      | Body            | Client's vTiger CRM URL
    * elementType    | Body            | Any of types returned by ListTypes endpoint (Contacts, Groups, etc.)
    * Authorization  | Headers         | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "elementType": "Contacts"
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "label": "Contacts",
    *       "name": "Contacts",
    *       "createable": true,
    *       "updateable": true,
    *       "deleteable": true,
    *       "retrieveable": true,
    *       "fields": [
    *         {
    *           "name": "account_id",
    *           "label": "Organization Name",
    *           "mandatory": false,
    *           "type": {
    *               "refersTo": [],
    *               "name": "reference"
    *           },
    *           "isunique": false,
    *           "nullable": true,
    *           "editable": true,
    *           "default": ""
    *         },
    *         {
    *           "name": "assistant",
    *           "label": "Assistant",
    *           "mandatory": false,
    *           "type": {
    *               "name": "string"
    *           },
    *           "isunique": false,
    *           "nullable": true,
    *           "editable": true,
    *           "default": ""
    *         },
    *         .....
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function Describe(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $elementType = $args['elementType'];

        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];

        $api = new RestAccess();
        $result = $api->get($apiUrl, array("operation" => "describe", "sessionName" => $this->session_id, "elementType" => $elementType));
        unset($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        return array(ApiConstants::KEY_RAW_RESPONSE => $response);
    }

   /* GetContactInfo endpoint
    * @request POST api/v1/Vtiger/GetContactInfo
    * @parameters
    * Name                  | Placement       | Description
    * ----------------------|-----------------|----------------------------------------------------------
    * vTigerUrl             | Body            | Client's vTiger CRM URL
    * phoneNumber           | Body            | Phone number to look for
    * searchPhoneTypes      | Body            | Array of phonetype fields to be searched for phoneNumber
    * mandatoryFields       | Body            | Array of mandatory fields whose value need to be get
    * minNumOfMatchedDigits | Body            | Minimal number of digits to match the phone number
    * exactMatch            | Body            | Boolean: if true, exact match of complete phone number required
    * Authorization         | Headers         | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "phoneNumber": "+37491900046"
    *   "searchPhoneTypes": [
    *      "mobile",
    *      "phone",
    *      "homephone"
    *   ],
    *   "mandatoryFields": [
    *      "firstname",
    *      "lastname",
    *      "mobile",
    *      "assigned_user_id"
    *   ]
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": [
    *       {
    *         "firstname": "Armen",
    *         "lastname": "Movsisyan",
    *         "email": "armen.movsisyan@epygiarm.am",
    *         "assistant": "",
    *         "assigned_user_id": "19x6",
    *         "id": "12x6"
    *       },
    *       ...........
    *     ]
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetContactInfo(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $phoneNumber = $args['phoneNumber'];
        $searchPhoneTypes = $args['searchPhoneTypes'];
        $mandatoryFields = $args['mandatoryFields'];
        $minNumOfMatchedDigits = $args['minNumOfMatchedDigits'];
        $exactMatch = $args['exactMatch'];

        if (!count($searchPhoneTypes)) {
            $response  = array (
                self::KEY_SUCCESS => false,
                self::KEY_ERROR => array (
                    self::KEY_CODE => self::ARGUMENT_MAY_NOT_BE_EMPTY_ERR_CODE,
                    self::KEY_MESSAGE => sprintf($this->ARGUMENT_MAY_NOT_BE_EMPTY_ERR_MSG, 'searchPhoneTypes')
                )
            );

            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                array(self::KEY_VTIGER_CALL_FAIL => $response), $response);
        }

        if ($minNumOfMatchedDigits > strlen($phoneNumber)) {
            $response  = array (
                self::KEY_SUCCESS => true,
                self::KEY_RESULT => array ()
            );
            return $response;
        }

        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];

        $str_where = array();
        $str_what = array("firstname", "lastname", "email", "description");
        foreach ($searchPhoneTypes as $phonetype) {
            if ($exactMatch === true) {
                $str_where[] = "$phonetype = '$phoneNumber'";
            } else {
                $str_where[] = "$phonetype like '%$phoneNumber%'";
            }
            $str_what[] = $phonetype;
        }
        foreach ($mandatoryFields as $fld) {
            $str_what[] = $fld;
        }
        $str_what = array_unique($str_what);
        $where = implode(" or ", $str_where);
        $what = implode(", ", $str_what);
        $query = "SELECT $what FROM Contacts where $where;";
        $api = new RestAccess();
        $result = $api->get($apiUrl, array("operation" => "query", "sessionName" => $this->session_id, "query" => $query));
        unset($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        return array(ApiConstants::KEY_RAW_RESPONSE => $response);
    }

   /* GetResourceInfo endpoint
    * @request POST api/v1/Vtiger/GetResourceInfo
    * @parameters
    * Name                       | Placement     | Description
    * ---------------------------|---------------|---------------------------------------------------------------------
    * vTigerUrl                  | Body          | Client's vTiger CRM URL
    * elementTypes               | Body          | Array of any types returned by ListTypes endpoint (Contacts, Groups, etc.)
    * Authorization              | Headers       | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "elementTypes": [
    *      "Contacts",
    *      "Groups"
    *   ]
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contacts": {
    *           "success": true
    *       },
    *       "Groups": {
    *           "code": "ACCESS_DENIED",
    *           "message": "Permission to perform the operation is denied for name : Groups",
    *           "success": false
    *       }
    *     }
    *   }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetResourceInfo(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $elementTypes = $args['elementTypes'];

        if (!count($elementTypes)) {
            $response  = array (
                self::KEY_SUCCESS => false,
                self::KEY_ERROR => array (
                    self::KEY_CODE => self::ARGUMENT_MAY_NOT_BE_EMPTY_ERR_CODE,
                    self::KEY_MESSAGE => sprintf($this->ARGUMENT_MAY_NOT_BE_EMPTY_ERR_MSG, 'elementTypes')
                )
            );

            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                array(self::KEY_VTIGER_CALL_FAIL => $response), $response);
        }

        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];

        $info = array();
    
        $generic_success = false;
        foreach ($elementTypes as $elementType) {
            $api = new RestAccess();
            $result = $api->get($apiUrl, array("operation" => "describe", "sessionName" => $this->session_id, "elementType" => $elementType));
            unset($api);
    
            if ($result[ApiConstants::KEY_ERROR] || (!$result[ApiConstants::KEY_RESPONSE][ApiConstants::KEY_SUCCESS] && $result[ApiConstants::KEY_RESPONSE][ApiConstants::KEY_ERROR][self::KEY_CODE] != "ACCESS_DENIED")) {
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
            
            if (!$result[ApiConstants::KEY_RESPONSE][ApiConstants::KEY_SUCCESS]) {
                $info[$elementType] = $result[ApiConstants::KEY_RESPONSE][self::KEY_ERROR];
                $info[$elementType][self::KEY_SUCCESS] = false;
            } else {
                $info[$elementType][self::KEY_CREATEABLE] = ($result[ApiConstants::KEY_RESPONSE][self::KEY_RESULT][self::KEY_CREATEABLE]) ? true : false;
                $info[$elementType][self::KEY_DELETEABLE] = ($result[ApiConstants::KEY_RESPONSE][self::KEY_RESULT][self::KEY_DELETEABLE]) ? true : false;
                $info[$elementType][self::KEY_UPDATEABLE] = ($result[ApiConstants::KEY_RESPONSE][self::KEY_RESULT][self::KEY_UPDATEABLE]) ? true : false;
                $info[$elementType][self::KEY_SUCCESS] = true;
                $generic_success = true;
            }
        }

        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (
                self::KEY_SUCCESS => $generic_success,
                self::KEY_RESULT => $info
            )
        );

        return $response;
    }

   /* GetMandatoryFields endpoint
    * @request POST api/v1/Vtiger/GetMandatoryFields
    * @parameters
    * Name                       | Placement     | Description
    * ---------------------------|---------------|---------------------------------------------------------------------
    * vTigerUrl                  | Body          | Client's vTiger CRM URL
    * elementTypes               | Body          | Array of any types returned by ListTypes endpoint (Contacts, Groups, etc.)
    * userDefinedMandatoryFields | Body          | Array of user defined mandatory fields
    * Authorization              | Headers       | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "elementTypes": [
    *      "Contacts"
    *      "Groups"
    *   ],
    *   "userDefinedMandatoryFields": {
    *      "Contacts": [
    *         "contactstatus",
    *         "happiness_rating"
    *      ],
    *      "Groups": [
    *         "groupname"
    *      ],
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contacts": [
    *         {
    *           "name": "lastname",
    *           "label": "Last Name",
    *           "type": "string",
    *           "isunique": false,
    *           "default": "",
    *           "picklistValues": []
    *         },
    *         {
    *           "name": "assigned_user_id",
    *           "label": "Assigned To",
    *           "type": "owner",
    *           "isunique": false,
    *           "default": "",
    *           "picklistValues": []
    *         },
    *         {
    *           "name": "contacttype",
    *           "label": "Contact Type",
    *           "type": "picklist",
    *           "isunique": false,
    *           "default": "Lead",
    *           "picklistValues": [
    *               {
    *                 "label": "Lead",
    *                 "value": "Lead"
    *               },
    *               {
    *                 "label": "Customer",
    *                 "value": "Customer"
    *               }
    *           ]
    *         }
    *       ],
    *       "Groups": [
    *         {
    *           "name": "groupname",
    *           "label": "groupname",
    *           "type": "string",
    *           "isunique": null,
    *           "default": null,
    *           "picklistValues": []
    *         }
    *       ],
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetMandatoryFields(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $elementTypes = $args['elementTypes'];
        $userDefinedMandatoryFields = ($args['userDefinedMandatoryFields'] ? $args['userDefinedMandatoryFields'] : array());
    
        if (!count($elementTypes)) {
            $response = array (
                ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true),
                self::KEY_RESULT => array()
            );
            return $response;
        }
        
        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];
    
        foreach ($elementTypes as $elementType) {
            $api = new RestAccess();
            $result = $api->get($apiUrl, array("operation" => "describe", "sessionName" => $this->session_id, "elementType" => $elementType));
            unset($api);
    
            if ($result[ApiConstants::KEY_ERROR]) {
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
    
            $response = $result[ApiConstants::KEY_RESPONSE];
    
            if (!$response[self::KEY_SUCCESS]) {
                throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
    
            $fields = $response[self::KEY_RESULT][self::KEY_FIELDS];
            $mandatory_fields[$elementType] = array();
            foreach ($fields as $fld) {
                if ($fld['mandatory'] || (in_array($fld["name"], $userDefinedMandatoryFields[$elementType]) === true)) {
                    $picklistValues = array();
                    if ($fld['type']['picklistValues']) {
                        $picklistValues = $fld['type']['picklistValues'];
                    }
                    $mandatory_fields[$elementType][] = array(
                        'name' => $fld['name'],
                        'label' => $fld['label'],
                        'type' => $fld['type']['name'],
                        'isunique' => $fld['isunique'],
                        'default' => $fld['default'],
                        'picklistValues' => $picklistValues
                    );
                }
            }
        }
        
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (
                self::KEY_SUCCESS => true,
                self::KEY_RESULT => $mandatory_fields
            )
        );
        return $response;
    }

   /* GetPhonetypeFields endpoint
    * @request POST api/v1/Vtiger/GetPhonetypeFields
    * @parameters
    * Name           | Placement       | Description
    * ---------------|-----------------|----------------------------------------------------------------------
    * vTigerUrl      | Body            | Client's vTiger CRM URL
    * elementTypes   | Body            | Array of any types returned by ListTypes endpoint (Contacts, Groups, etc.)
    * Authorization  | Headers         | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "elementTypes": [
    *      "Contacts"
    *      "Groups"
    *   ]
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contacts": [
    *         {
    *           "name": "assistantphone",
    *           "label": "Assistant Phone",
    *           "editable": true
    *         },
    *         {
    *           "name": "homephone",
    *           "label": "Home Phone",
    *           "editable": true
    *         }
    *       ],
    *       "Groups": [
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetPhonetypeFields(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $elementTypes = $args['elementTypes'];
    
        if (!count($elementTypes)) {
            $response = array (
                ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true),
                self::KEY_RESULT => array()
            );
            return $response;
        }
    
        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];
    
        foreach ($elementTypes as $elementType) {
            $api = new RestAccess();
            $result = $api->get($apiUrl, array("operation" => "describe", "sessionName" => $this->session_id, "elementType" => $elementType));
            unset($api);
    
            if ($result[ApiConstants::KEY_ERROR]) {
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
    
            $response = $result[ApiConstants::KEY_RESPONSE];
    
            if (!$response[self::KEY_SUCCESS]) {
                throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
    
            $fields = $response[self::KEY_RESULT][self::KEY_FIELDS];
            $phonetype_fields[$elementType] = array();
            foreach ($fields as $fld) {
                if ($fld['type']['name'] == 'phone') {
                    $phonetype_fields[$elementType][] = array(
                        'name' => $fld['name'],
                        'label' => $fld['label'],
                        'editable' => $fld['editable'],
                    );
                }
            }
        }
        
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (
                self::KEY_SUCCESS => true,
                self::KEY_RESULT => $phonetype_fields
            )
        );
        return $response;
    }

    /* GetListOfFields endpoint
    * @request POST api/v1/Vtiger/GetListOfFields
    * @parameters
    * Name                       | Placement     | Description
    * ---------------------------|---------------|---------------------------------------------------------------------
    * vTigerUrl                  | Body          | Client's vTiger CRM URL
    * elementTypes               | Body          | Array of any types returned by ListTypes endpoint (Contacts, Groups, etc.)
    * Authorization              | Headers       | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "elementTypes": [
    *      "Contacts"
    *      "Groups"
    *   ]
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contacts": [
    *          "salutationtype",
    *          "firstname",
    *          "lastname",
    *          "email",
    *          "phone",
    *          .....
    *       ],
    *       "Groups": [
    *          .....
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetListOfFields(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $elementTypes = $args['elementTypes'];
    
        if (!count($elementTypes)) {
            $response = array (
                ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true),
                self::KEY_RESULT => array()
            );
            return $response;
        }

        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];
    
        foreach ($elementTypes as $elementType) {
            $api = new RestAccess();
            $result = $api->get($apiUrl, array("operation" => "describe", "sessionName" => $this->session_id, "elementType" => $elementType));
            unset($api);
    
            if ($result[ApiConstants::KEY_ERROR]) {
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
    
            $response = $result[ApiConstants::KEY_RESPONSE];
    
            if (!$response[self::KEY_SUCCESS]) {
                throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
            }
    
            $fields = $response[self::KEY_RESULT][self::KEY_FIELDS];
            $list_of_fields[$elementType] = array();
            foreach ($fields as $fld) {
                if (($fld['type']['name'] == 'image') || ($fld['type']['name'] == "reference") || ($fld['type']['name'] == "owner") || ($fld['type']['name'] == "autogenerated")) {
                    //skip for now 'image', 'reference', 'owner' and 'autogenerated' types
                    continue;
                }
                $list_of_fields[$elementType][] = $fld['name'];
            }
        }
        
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (
                self::KEY_SUCCESS => true,
                self::KEY_RESULT => $list_of_fields
            )
        );
        return $response;
    }

   /* CreateContact endpoint (Creates/Updates Contact)
    * @request POST api/v1/Vtiger/CreateContact
    * @parameters
    * Name                    | Placement | Description
    * ------------------------|-----------|-------------------------------------------------------------------------------
    * vTigerUrl               | Body      | Client's vTiger CRM URL
    * vTigerUserName          | Body      | Login User name for client's vTiger CRM (this is default value for assignedTo)
    * firstname               | Body      | Contact First Name
    * lasttname               | Body      | Contact Last Name
    * mobile                  | Body      | Contact Mobile Phone
    * phone                   | Body      | Contact Office Phone
    * homephone               | Body      | Contact Home Phone
    * assistantphone          | Body      | Contact Assistant Phone
    * otherphone              | Body      | Contact Secondary Phone
    * fax                     | Body      | Contact Fax
    * email                   | Body      | Contact Email Address
    * description             | Body      | Contact Description
    * assignedToOrigin        | Body      | Origination of assigned_to field. One of UserName|GroupName (default 'UserName')
    * assignedTo              | Body      | "Assigned To" field value (depends on assignedToOrigin) (default logged in username)
    * assigned_user_id        | Body      | "Assigned To" field's usder id (default logged in user's id)
    * customManadatoryFields  | Body      | List of name/value pairs
    * id                      | Body      | In case if we do update the contact - id of the contact to be updated
    * Authorization           | Headers   | Session ID
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "vTigerUrl": "https://epygi.vtiger.com",
    *   "vTigerUserName": "armen@epygi.com",
    *   "firstname": "Armen",
    *   "lastname": "Movsisyan",
    *   "mobile": "091998877",
    *   "homephone": "556677",
    *   "phone": "+181812345678",
    *   "assistantphone": 181812345678,
    *   "email": "armen@movsisyan.com",
    *   "description": "Put some description here",
    *   "customManadatoryFields": [
    *     {
    *       "name": "contacttype",
    *       "value": "Lead"
    *     },
    *     {
    *       "name": "contactstatus",
    *       "value": "Warm"
    *     }
    *   ]
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": [
    *       {
    *         "salutationtype": "",
    *         "firstname": "Armen",
    *         "lastname": "Movsisyan",
    *         "email": "armen@epygi.com",
    *         "phone": "+13657099135",
    *         "mobile": "091900046",
    *         "homephone": "551606",
    *         "assistant": "",
    *         "assigned_user_id": "19x6",
    *         "id": "12x6"
    *         ..........
    *       }
    *     ]
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function CreateContact(array $args) {
        $apiUrl = rtrim($args['vTigerUrl'],'/') . '/' . self::API_URL_PATH;
        $vTigerUserName = $args['vTigerUserName'];
        $assignedToOrigin = $args['assignedToOrigin'];
        $assignedToName = ($args['assignedToName']) ? trim($args['assignedToName']) : $vTigerUserName;
        $assigned_user_id = trim($args['assigned_user_id']);
        $customManadatoryFields = $args['customManadatoryFields'];
        $id = $args['id'];

        $this->session_id = $_SERVER['HTTP_AUTHORIZATION'];

        if ($assigned_user_id) {
            $this->assigned_to_id = $assigned_user_id;
        } else {
            $this->getAssignToID($apiUrl, $assignedToOrigin, $assignedToName);
        }
    
        $headers = array ('Content-Type' => 'application/x-www-form-urlencoded');
        $api = new RestAccess($headers);

        $element_array = array ();
        if ($args["firstname"] != '') {
            $element_array["firstname"] = trim($args['firstname']);
        }
        if ($args["lastname"] != '') {
            $element_array["lastname"] = trim($args['lastname']);
        }
        if ($args["email"] != '') {
            $element_array["email"] = trim($args['email']);
        }
        if ($args["description"] != '') {
            $element_array["description"] = trim($args['description']);
        }
        if ($this->assigned_to_id) {
            $element_array["assigned_user_id"] = $this->assigned_to_id;
        }
        if ($args["mobile"] != '') {
            $element_array["mobile"] = trim($args['mobile']);
        }
        if ($args["phone"] != '') {
            $element_array["phone"] = trim($args['phone']);
        }
        if ($args["homephone"] != '') {
            $element_array["homephone"] = trim($args['homephone']);
        }
        if ($args["assistantphone"] != '') {
            $element_array["assistantphone"] = trim($args['assistantphone']);
        }
        if ($args["otherphone"] != '') {
            $element_array["otherphone"] = trim($args['otherphone']);
        }
        if ($args["fax"] != '') {
            $element_array["fax"] = trim($args['fax']);
        }
        if ($id) {
            $element_array["id"] = $id;
        }

        foreach ($customManadatoryFields as $fld) {
            $element_array[$fld["name"]] = $fld["value"];
        }

        $element_json = json_encode($element_array);

        if ($id) {
            $result = $api->post($apiUrl, array("operation"=> "update", "sessionName" => $this->session_id, "element" => $element_json));
        } else {
            $result = $api->post($apiUrl, array("operation"=> "create", "sessionName" => $this->session_id, "element" => $element_json, "elementType" => "Contacts"));
        }
        unset ($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        return array(ApiConstants::KEY_RAW_RESPONSE => $response);
    }

    // Consecutive call for vTiger API Challenge and Login to set $this->session_id
    private function getSessionIDinternal($apiUrl, $vTigerUserName, $vTigerUserAccessKey) {
        $this->challenge($apiUrl, $vTigerUserName);
        $response = $this->login($apiUrl, $vTigerUserName, $vTigerUserAccessKey);
        return $response;
    }

    // Call for vTger API Login endpoint;
    // Gets sessionName to be used for other requests
    private function login($apiUrl, $vTigerUserName, $vTigerUserAccessKey) {
        $headers = array ('Content-Type' => 'application/x-www-form-urlencoded');
        $api = new RestAccess($headers);
        $accessKey = MD5($this->token . $vTigerUserAccessKey);
        $result = $api->post($apiUrl, array("operation"=> "login", "username" => $vTigerUserName,	"accessKey" => $accessKey));
        unset ($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $this->session_id = $response[self::KEY_RESULT][self::KEY_SESSION_NAME];

        return $response;
    }

    // Call for vTger API Challenge endpoint;
    // Gets Challenge token to be used for login
    private function challenge($apiUrl, $vTigerUserName) {
        $api = new RestAccess();
        $result = $api->get($apiUrl, array("operation" => "getchallenge", "username" => $vTigerUserName));
        unset ($api);
        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $this->token = $response[self::KEY_RESULT][self::KEY_TOKEN];

        return $response;
    }

    private function getAssignToID($apiUrl, $assignedToOrigin, $assignedTo) {
        if ($assignedToOrigin === 'UserName') {
            $type = 'Users';
            $field = 'user_name';
        } else {
            $type = 'Groups';
            $field = 'groupname';
        }
        $query = "SELECT id FROM $type where $field='$assignedTo';";
        $api = new RestAccess();
        $result = $api->get($apiUrl, array("operation" => "query", "sessionName" => $this->session_id, "query" => $query));
        unset ($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        if (!$response[self::KEY_SUCCESS]) {
            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $result[ApiConstants::KEY_RESPONSE]), $result[ApiConstants::KEY_RESPONSE]);
        }

        $this->assigned_to_id = $result[ApiConstants::KEY_RESPONSE][self::KEY_RESULT][0][self::KEY_ID];

        if (!$this->assigned_to_id) {
            $response  = array (
                self::KEY_SUCCESS => false,
                self::KEY_ERROR => array (
                    self::KEY_CODE => self::ASSIGNED_TO_ID_NOT_FOUND_ERR_CODE,
                    self::KEY_MESSAGE => sprintf($this->ASSIGNED_TO_ID_NOT_FOUND_ERR_MSG, $assignedToOrigin, $assignedTo)
                )
            );

            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_VTIGER_CALL_FAIL => $response), $response);
        }

        return $response;
    }
}
?>
