<?php
namespace App\Doc;

use Epygi;
use Exception;
use Mvc;

class RouterStudy
{
    private $cgi_list = array();
    private $children = array();
    private $full_list = array();
    private $show = null;

    public function __construct()
    {
        $this->cgi_list = include ROOT_DIR . '/assets/config/cgi-list.php';

        // iterate through all cgilist routes and generate parent/child relationships
        foreach ($this->cgi_list as $cginame => $data) {

            // who is this route's parent?
            $parent = isset($data['parent']) ? $data['parent'] : false;

            // make sure the parent array index has been created
            if (!isset($this->children[$parent])) {
                $this->children[$parent] = array();
            }

            // add this child to our parent array
            $this->children[$parent][] = $cginame;
        }

        // save the full list for recursive lookups
        $this->full_list = $this->cgi_list;

        /**
         * initialize the 'show' variable so we can see stuff
         */
        // children of root
        $children = $this->getChildren('ROOT');

        // only show this node
        $this->show = Mvc\Param::request('show');

        // show all if not a known child of root
        if (!isset($children[$this->show])) {
            $this->show = 'all';
        }
    }

    public function debugLeftovers()
    {
        // nothing to do
        if ($this->show !== 'all') {
            return;
        }

        // debug left overs
        unset($this->children['ROOT']);
        dump($this->children);
    }

    public function extractChildren(&$data, $parent, $level = 1)
    {
        // save this route
        $route = $this->cgi_list[$parent];
        $route['level'] = $level;
        if (!isset($route['label'])) {
            throw new Exception ('missing label for router entry: ' . print_r($route, true));
        }
        $data[$parent] = $route;

        // extract all the children of this route
        $children = $this->getChildren($parent);
        foreach ($children as $child => $menu) {
            $this->extractChildren($data, $child, $level + 1);
        }

        // after drawing a parent, remove it
        unset($this->cgi_list[$parent]);
        unset($this->children[$parent]);
    }

    public function getAcl($cgi_name)
    {
        return Mvc\Router::getInstance()->getAcl($cgi_name);
    }

    public function getAttributeRecursive($attr_name, $cgi_name)
    {
        // invalid cgi name, stop searching
        if (!isset($this->full_list[$cgi_name])) {
            return false;
        }

        // if the attribute is set on this route, use it ... done
        if (isset($this->full_list[$cgi_name][$attr_name])) {
            return $this->full_list[$cgi_name][$attr_name];
        }

        // if we have a parent route, recursively check if the value was set there
        $parent = isset($this->full_list[$cgi_name]['parent']) ? $this->full_list[$cgi_name]['parent'] : false;
        if ($parent && $parent !== 'ROOT') {
            return $this->getAttributeRecursive($attr_name, $parent);
        }

        // attribute not found!
        return false;
    }

    /**
     * TODO: make sure this list of children has been restricted to only the menu items which match our ACL, ROLE,
     * and MENU settings.
     *
     * @param $parent - parent route ("ROOT" for top level)
     *
     * @return array - array of child routes
     */
    public function getChildren($parent)
    {
        // we have all our children indexed already ... how convenient
        $children = isset($this->children[$parent]) ? $this->children[$parent] : array();

        // collect route data for each of these children
        $data = array();
        foreach ($children as $child) {
            if (!isset($this->cgi_list[$child])) {
                continue;
            }
            $route = $this->cgi_list[$child];
            $data[$child] = $route;
        }
        return $data;
    }

    public function getHelp($child)
    {
        // this is a menu, it doesn't have help
        if (!$this->needHelp($child)) {
            return '';
        }

        // the help file is missing
        if (!$this->hasHelp($child)) {
            return '';
        }

        // here is the help link
        return '/help/' . $child;
    }

    public function hasHelp($child)
    {
        return (boolean) Mvc\Application::findAssetFile('view', 'help/' . $child);
    }

    public function needHelp($child)
    {
        // if this is just a menu, we don't really need help
        return preg_match('#\.#', $child) ? false : true;
    }
}