<?php
namespace Epygi\Db;

use Epygi\Db\Statement\Helper;
use PDO;

class Statement
{
    // lines of sql
    private $_bind_position = 0;

    // named bind values
    private $_filter_position = 0;

    // used to translate positional placeholders into named placeholders
    private $_named = array();
    private $_sql = array();
    private $_sql_position = 0;

    function __toString()
    {
        return $this->getDebug();
    }

    /**
     * Used to assign a named parameter.
     *
     * @param $name
     * @param $value
     * @param int $type
     *
     * @return $this
     */
    public function bind($name, $value, $type = PDO::PARAM_STR)
    {
        $this->_named[$name] = array(
            'type' => $type,
            'value' => $value
        );

        // enable chaining
        return $this;
    }

    /**
     *
     */
    public function dump()
    {
        dump($this->getDebug());
        edump($this->getDebug());
    }

    /**
     * @param PDO $PDO
     *
     * @return \PDOStatement
     */
    public function execute(PDO $PDO)
    {
        // prepare the SQL
        $sql = join(' ', $this->_sql);
        $stmt = $PDO->prepare($sql);

        // bind named parameters
        foreach ($this->_named as $name => $sval) {
            switch ($sval['type']) {
                case PDO::PARAM_BOOL :
                    $stmt->bindValue($name, (boolean) $sval['value'], $sval['type']);
                    break;

                case PDO::PARAM_NULL :
                    $stmt->bindValue($name, null);
                    break;

                case PDO::PARAM_INT :
                    $stmt->bindValue($name, $sval['value'], $sval['type']);
                    break;

                case PDO::PARAM_STR :
                default :
                    $stmt->bindValue($name, (string) $sval['value'], $sval['type']);
                    break;
            }
        }

        $stmt->execute();
        return $stmt;
    }

    /**
     * Developers need to be able to cut and paste SQL strings into their database tools and need the values to be
     * merged into their positions. This function achieves that but doesn't guarantee that the query is exactly what
     * gets sent through PDO and the prepared statements.
     * @return mixed|string
     */
    public function getDebug()
    {
        // merge sql together
        $sql = join("\n", $this->_sql);

        // replace positioned placeholders with named placeholders (first value)
        $sql = preg_replace_callback(
            '/:[a-z0-9_]+/m',
            array(
                $this,
                'placeholderFill'
            ),
            $sql
        );

        return $sql;
    }

    /**
     * Without linebreaks, return the SQL string.
     * @return string
     */
    public function getDebugLine()
    {
        $sql = join(' ', $this->_sql);

        // replace positioned placeholders with named placeholders (first value)
        $sql = preg_replace_callback(
            '/:[a-z0-9_]+/m',
            array(
                $this,
                'placeholderFill'
            ),
            $sql
        );

        return $sql;
    }

    /**
     * @param bool $type
     *
     * @return string
     */
    private function getNextName($type = false)
    {
        switch ($type) {
            case 'filter' :
                // filter
                return sprintf(':filter%d', $this->_filter_position++);

            case 'sql' :
                // sql statement syntax
                return sprintf(':pos%d', $this->_sql_position++);

            case 'bind' :
            default :
                // bind/filling values
                return sprintf(':pos%d', $this->_bind_position++);
        }
    }

    /**
     * Takes an array of values and ensures all values are integers and that the integers are unique.  The values are
     * joined with commas and are intended to be used with 'IN (...)' clauses.  If no values exist, the 'default'
     * string is returned.
     *
     * @param $data
     * @param string $default
     *
     * @return string
     */
    public static function intArray($data, $default = '0')
    {
        // make unique integer array
        $nums = array();
        foreach ((array) $data as $value) {
            $nums[(int) $value] = true;
        }
        $nums = array_keys($nums);

        // turn into a string
        $result = join(', ', $nums);
        return $result ? $result : $default;
    }

    /**
     * Helper function to build a LIMIT and OFFSET clause onto the query.
     *
     * @param Helper $HELPER
     *
     * @return $this
     */
    public function limitOffset(Helper $HELPER)
    {
        $this->sql($HELPER->getSqlLimitOffset());

        // enable chaining
        return $this;
    }

    /**
     * Dump the query to our logs.
     */
    public function logDump()
    {
        edump($this->getDebug());
    }

    private static function match()
    {
        // first argument is the needle, all other arguments are the haystack
        $haystack = func_get_args();
        $needle = array_shift($haystack);

        // search for our needle in the haystack
        return in_array($needle, $haystack, true);
    }

    /**
     * Helper function to build an ORDER BY clause onto the query. All parameters are treated as "field",
     * "direction" pairs unless a "Helper" object is passed in, then the order defined on that object
     * is inserted in place.  The clause is only added if there exists a column and direction provided after all args
     * are processed.  Duplicates field names silently removed.  examples:
     * $SQL->orderBy('first', 'ASC', $HELPER);
     * $SQL->orderBy($HELPER, 'created', 'DESC');
     * $SQL->orderBy('created', 'DESC');
     * $SQL->orderBy('first', 'DESC', $HELPER, 'created', 'DESC');
     * @return Statement|string
     */
    public function orderBy()
    {
        // args
        $args = func_get_args();

        // build array of order by args key=field, value=direction
        $orders = array();
        while (count($args)) {
            // check next arg
            $arg = array_shift($args);

            // we found a NULL object (skip it)
            if ($arg === null) {
                continue;
            }
            // we found the HELPER object
            elseif ($arg instanceof Helper) {
                $HELPER = $arg;
                $field = $HELPER->getOrderColumn();
                $dir = $HELPER->getOrderDir();
            }
            // must be COL, DIR pair
            else {
                // read next value from args
                $field = $arg;
                $dir = array_shift($args);
            }

            // allow different ways of describing sort direction
            if ($dir === true || $dir === 1) {
                $dir = 'ASC';
            }
            elseif ($dir === false || $dir === 0 || $dir === -1) {
                $dir = 'DESC';
            }

            // add additional order constraints (use assoc array to remove dupes)
            if ($field && !isset ($orders[$field])) {
                $orders[$field] = $dir;
            }
        }

        // nothing to order by
        if (!count($orders)) {
            return '';
        }

        // build order SQL
        $order = '';
        foreach ($orders as $col => $dir) {
            $order .= ", $col $dir";
        }

        // return full SQL order by clause
        return $this->sql('ORDER BY ' . trim($order, ' ,'));
    }

    /**
     * @param $matches
     *
     * @return int|string
     */
    private function placeholderFill($matches)
    {
        $key = $matches[0];

        // can't file this param
        if (!isset ($this->_named[$key])) {
            return $key;
        }

        // here is the param
        $sval = $this->_named[$key];

        switch ($sval['type']) {
            case PDO::PARAM_BOOL :
                return $sval['value'] ? 'TRUE' : 'FALSE';

            case PDO::PARAM_NULL :
                return 'NULL';

            case PDO::PARAM_INT :
                return (int) $sval['value'];

            case PDO::PARAM_STR :
            default :
                // TODO: need something like mysql_escape_string to escape this value when debugging
                return "'" . $sval['value'] . "'";
        }
    }

    /**
     * @param $matches
     *
     * @return string
     */
    private function placeholderGetName($matches)
    {
        return $this->getNextName('sql');
    }

    /**
     * @param PDO $pdo
     *
     * @return \PDOStatement
     */
    public function prepare(PDO $pdo)
    {
        // prepare the SQL
        $sql = join(' ', $this->_sql);
        return $pdo->prepare($sql);
    }

    /**
     * Used to assign a "positional" parameter which just ends up getting translated to a named parameter magically.
     *
     * @param $value
     * @param int $type
     *
     * @return Statement
     */
    public function set($value, $type = PDO::PARAM_STR)
    {
        $name = $this->getNextName();
        return $this->bind($name, $value, $type);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setBigint($value)
    {
        return $this->setInt($value);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setBoolean($value)
    {
        $value = self::test($value);
        return $this->set($value, PDO::PARAM_BOOL);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setFilter($value)
    {
        return $this->bind('filter', $value, PDO::PARAM_STR);
    }

    /**
     * @param $value
     * @param int $decimals
     *
     * @return Statement
     */
    public function setFloat($value, $decimals = 3)
    {
        $format = sprintf('%%0.%df', $decimals);
        return $this->set(sprintf($format, $value));
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setGuid($value)
    {
        $value = trim($value);

        // use NULL
        if (strlen($value) !== 32) {
            return $this->set(null, PDO::PARAM_NULL);
        }

        // use string value
        return $this->set($value, PDO::PARAM_STR);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setInt($value)
    {
        $value = sprintf('%d', $value);
        // $value = number_format($value, 0, '', ''); // sprintf('%d', $value);
        return $this->set($value, PDO::PARAM_INT);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setMoney($value)
    {
        return $this->set(sprintf('%0.2f', $value));
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setn($value)
    {
        // use NULL
        if (!$value) {
            return $this->set(null, PDO::PARAM_NULL);
        }

        // use string value
        return $this->set($value, PDO::PARAM_STR);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setnBigint($value)
    {
        return $this->setnInt($value);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setnBoolean($value)
    {
        // use NULL
        if (!$value) {
            return $this->set(null, PDO::PARAM_NULL);
        }

        // use boolean value
        return $this->setBoolean($value);
    }

    /**
     * @param $value
     *
     * @return Statement
     */
    public function setnInt($value)
    {
        // use NULL
        if (!$value) {
            return $this->set(null, PDO::PARAM_NULL);
        }

        // use int value
        return $this->setInt($value);
    }

    /**
     * @param $text
     *
     * @return $this
     */
    public function sql($text)
    {
        // replace positioned placeholders with named placeholders (first value)
        $text = preg_replace_callback(
            '/\?/m',
            array(
                $this,
                'placeholderGetName'
            ),
            $text
        );

        // just add the text as-is
        if (func_num_args() === 1) {
            $this->_sql[] = $text;
        }
        // treat as sprintf statement
        else {
            $args = func_get_args();
            $args[0] = $text;
            $this->_sql[] = call_user_func_array('sprintf', $args);
        }

        // enable chaining
        return $this;
    }

    /**
     * @param $text
     * @param $args
     *
     * @return $this
     */
    public function sqlIn($text, $args)
    {
        // no array elements
        if (!is_array($args)) {
            $args = func_get_args();
            $text = array_shift($args);
        }

        // no values
        if (!count($args)) {
            return $this->sql($text, 'NULL');
        }

        // render placeholders
        $placeholders = trim(str_repeat('?, ', count($args)), ', ');
        $this->sql(sprintf($text, $placeholders));

        // bind each argument
        foreach ($args as $arg) {
            $this->setn($arg);
        }

        // chaining
        return $this;
    }

    /**
     * Takes an array of values and encodes them as an array of strings (for MySQL only).
     *
     * @param $values
     * @param string $default
     *
     * @return string
     */
    public static function stringArray($values, $default = 'NULL')
    {
        // no array elements
        if (!is_array($values) || !count($values)) {
            return $default;
        }

        // join array elements
        // $values = array_map('mysql_escape_string', $values);
        return "'" . join("', '", $values) . "'";
    }

    private static function test($value)
    {
        if (is_scalar($value)) {
            // true equivalents
            if (self::match($value, 'yes', 'true', 't', 'on', true)) {
                return true;
            }

            // false equivalents
            if (self::match($value, 'no', 'false', 'f', 'off', false)) {
                return false;
            }
        }

        // cast it as a boolean
        return (boolean) $value;
    }

    /**
     * Helper function to build a WHERE clause onto the query.  The clause is only added if there exists a filter and
     * column provided by the Helper object.
     *
     * @param Helper $HELPER
     * @param string $where_word
     *
     * @return $this
     */
    public function where(Helper $HELPER, $where_word = 'WHERE')
    {
        // get the filters
        $filters = $HELPER->getFilters();

        // no filters defined
        if (!count($filters)) {
            return '';
        }

        // process each filter and build SQL
        $filter_index = 1;
        foreach ($filters as $filter) {
            // read filter settings
            $field = $filter['field'];
            $value = $filter['value'];
            $type = $filter['type'];

            // apply filter based on type
            switch ($type) {
                case 'boolean' :
                    // cast as a number (digits only)
                    $value = self::test($value);
                    $line = sprintf('%s = %s', $field, $value ? 'TRUE' : 'FALSE');
                    break;

                case 'notequal' :
                    $fname = $this->getNextName('filter');
                    $this->bind($fname, $value);
                    $line = sprintf('%s <> %s', $field, $fname);
                    break;

                case 'number' :
                case 'numeric' :
                    // cast as a number (digits only)
                    $value = (int) preg_replace('/\D/', '', $value);
                    $line = sprintf('%s = %d', $field, $value);
                    break;

                case 'numericarray' :
                    // TODO: check array length, escape values, cast all as int
                    $line = sprintf('%s IN (%s)', $field, join(', ', $value));
                    break;

                case 'string' :
                    $fname = $this->getNextName('filter');
                    $this->bind($fname, $value);
                    $line = sprintf('%s = %s', $field, $fname);
                    break;

                case 'stringarray' :
                    $value = self::stringArray($value);
                    $line = sprintf('%s IN (%s)', $field, $value);
                    break;

                case 'substring' :
                default :
                    $fname = $this->getNextName('filter');
                    $this->bind($fname, $value);

                    $line = sprintf("%s LIKE CONCAT('%%', %s, '%%')", $field, $fname);
                    break;
            }

            // here is a line of sql
            $this->sql('%s %s', $where_word, $line);
            $where_word = 'AND';
        }

        // enable chaining
        return $this;
    }

    /**
     * This is a convenient function to write the debug query to a file.  It shouldn't be used by anyone other than
     * an admin doing debugging.
     *
     * @param $file_name
     */
    public function writeDebugFile($file_name)
    {
        file_put_contents($file_name, $this->getDebug());
    }
}