<?php
// This class uses SalesForce Lightning Platform REST API for third party integration
// see https://developer.salesforce.com/docs/atlas.en-us.api_rest.meta/api_rest/intro_what_is_rest_api.htm

class SalesForceModel_v1 extends BaseModel {

    const API_VERSION = 'v48.0';

    const KEY_SF_CALL_FAIL = "SalesForceCallFail";
    const KEY_SUCCESS = "success";
    const KEY_ERROR = "error";
    const KEY_ERROR_CODE = "errorCode";
    const KEY_RESULT = "result";
    const KEY_CODE = "code";
    const KEY_MESSAGE = "message";
    const KEY_ERROR_DESCRIPTION = "error_description";
    const KEY_RECORDS = "records";
    const KEY_ID = "Id";
    const KEY_FIELDS = "fields";
    const KEY_CREATEABLE = "createable";
    const KEY_DELETEABLE_REQW = "deletable";
    const KEY_DELETEABLE_RESP = "deleteable";
    const KEY_UPDATEABLE = "updateable";

    const OWNER_ID_NOT_FOUND_ERR_CODE = "OWNER_ID_NOT_FOUND";
    private $OWNER_ID_NOT_FOUND_ERR_MSG = "OwnerID for %s not found";

    const ARGUMENT_MAY_NOT_BE_EMPTY_ERR_CODE = "ARGUMENT_MAY_NOT_BE_EMPTY";
    private $ARGUMENT_MAY_NOT_BE_EMPTY_ERR_MSG = "%s argument may not be empty";
    
    private $ownerId;
    
    public function __construct(ApiRequest $apiRequestObj) {
        parent::__construct($apiRequestObj);
        $this->ownerId = null;
    }

   /* GetToken endpoint
    * @request POST api/v1/SalesForce/GetToken
    * @parameters
    * Name              | Placement | Description
    * ------------------|-----------|------------------------------------
    * clientId          | Body      | Connected App Client ID
    * clientSecret      | Body      | Connected App Client Secret
    * username          | Body      | SalesForce CRM User name
    * password          | Body      | SalesForce CRM User password
    * securityToken     | Body      | SalesForce CRM User securityToken
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "clientId": "3MVG9xB_D1giir9qSIzd765itrdsKgy4ze6jU.l0nKdhHXZs_.rjbgYPXAyJ",
    *   "clientSecret": "7FAA9012FF7B2002E765FFA09BCD343AB345D2A02E227BA2ED908FFABE21",
    *   "username": "armen@epygi.com"
    *   "password": "my_password"
    *   "securityToken": "oYyRsdFDFSFJrRW2NkGesdUwaEi"
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "access_token": "00D3z000001gpjc!gBAQJUKPO1tBVq_I4xN9hyjer98SawAo17.12vYyAANvd5dnbJYa_4i90f8oHmT_t237",
    *       "instance_url": "https://um1.salesforce.com",
    *       "id": "https://login.salesforce.com/id/00D3z0023401glkjDCF/0153z00234BHxJwAKW",
    *       "token_type": "Bearer",
    *       "issued_at": "1590486660732",
    *       "signature": "xVuVoQp3TXtF9skdTYjcnTs8VnmoWKoiuOzQQicMmQ="
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetToken(array $args) {
        $apiUrl = "https://login.salesforce.com/services/oauth2/token";

        $client_id = $args['clientId'];
        $client_secret = $args['clientSecret'];
        $username = $args['username'];
        $password = $args['password'];
        $securityToken = $args['securityToken'];

        $api = new RestAccess();
        $result = $api->post($apiUrl, array("grant_type" => "password", "client_id" => $client_id, "client_secret" => $client_secret, "username" => $username, "password" => $password.$securityToken));
        unset ($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            $converted_result = $this->convertError($result);
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];
        
        $converted_result = array(self::KEY_SUCCESS => true, self::KEY_RESULT => $response);
        return array(ApiConstants::KEY_RAW_RESPONSE => $converted_result);
    }

   /* ListObjects endpoint
    * @request POST api/v1/SalesForce/ListObjects
    * @parameters
    * Name                | Placement       | Description
    * --------------------|-----------------|-------------------------------------------
    * instanceUrl         | Body            | SalesForce API instance URL
    * Authorization       | Headers         | 'Bearer ' . access_token
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "instanceUrl": "https://um1.salesforce.com",
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "encoding": "UTF-8",
    *       "maxBatchSize": 200,
    *       "sobjects": [
    *          {
    *            "activateable": false,
    *            "createable": false,
    *            "custom": false,
    *            "customSetting": false,
    *            "deepCloneable": false,
    *            "deletable": false,
    *            "deprecatedAndHidden": false,
    *            "feedEnabled": false,
    *            "hasSubtypes": false,
    *            "isInterface": false,
    *            "isSubtype": false,
    *            "keyPrefix": null,
    *            "label": "Accepted Event Relation",
    *            "labelPlural": "Accepted Event Relations",
    *            "layoutable": false,
    *            "mergeable": false,
    *            "mruEnabled": false,
    *            "name": "AcceptedEventRelation",
    *            "queryable": true,
    *            "replicateable": false,
    *            "retrieveable": true,
    *            "searchable": false,
    *            "triggerable": false,
    *            "undeletable": false,
    *            "updateable": false,
    *            "urls": {
    *                "rowTemplate": "/services/data/v48.0/sobjects/AcceptedEventRelation/{ID}",
    *                "defaultValues": "/services/data/v48.0/sobjects/AcceptedEventRelation/defaultValues?recordTypeId&fields",
    *                "describe": "/services/data/v48.0/sobjects/AcceptedEventRelation/describe",
    *                "sobject": "/services/data/v48.0/sobjects/AcceptedEventRelation"
    *            }
    *          },
    *         {
    *          ...
    *         }
    *        ...
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function ListObjects(array $args) {
        $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects';

        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $api = new RestAccess(array("Authorization" => $access_token));
        $result = $api->get($apiUrl);
        unset($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            $converted_result = $this->convertError($result);
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        $converted_result = array(self::KEY_SUCCESS => true, self::KEY_RESULT => $response);
        return array(ApiConstants::KEY_RAW_RESPONSE => $converted_result);
    }

   /* Describe endpoint
    * @request POST api/v1/SalesForce/Describe
    * @parameters
    * Name           | Placement       | Description
    * ---------------|-----------------|----------------------------------------------------------------------
    * instanceUrl    | Body            | SalesForce API instance URL
    * sobject        | Body            | Any of objects returned by ListObjects endpoint (Contact, Lead, etc.)
    * Authorization  | Headers         | 'Bearer ' . access_token
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "instanceUrl": "https://um1.salesforce.com",
    *   "sobject": "Contact"
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "actionOverrides": [],
    *       "activateable": false,
    *       "childRelationships": [
    *          {
    *             "cascadeDelete": false,
    *             "childSObject": "AcceptedEventRelation",
    *             "deprecatedAndHidden": false,
    *             "field": "RelationId",
    *             "junctionIdListNames": [],
    *             "junctionReferenceTo": [],
    *             "relationshipName": "AcceptedEventRelations",
    *             "restrictedDelete": false
    *          },
    *          ....
    *       ],
    *       "compactLayoutable": true,
    *       "createable": true,
    *       "custom": false,
    *       "customSetting": false,
    *       "deepCloneable": false,
    *       "defaultImplementation": null,
    *       "deletable": true,
    *       "deprecatedAndHidden": false,
    *       "extendedBy": null,
    *       "extendsInterfaces": null,
    *       "feedEnabled": true,
    *       "fields": [
    *         {
    *           "aggregatable": true,
    *           "aiPredictionField": false,
    *           "autoNumber": false,
    *           "byteLength": 18,
    *           "calculated": false,
    *           "calculatedFormula": null,
    *           "cascadeDelete": false,
    *           "caseSensitive": false,
    *           "compoundFieldName": null,
    *           "controllerName": null,
    *           "createable": false,
    *           "custom": false,
    *           "defaultValue": null,
    *           "defaultValueFormula": null,
    *           "defaultedOnCreate": true,
    *           "dependentPicklist": false,
    *           "deprecatedAndHidden": false,
    *           "digits": 0,
    *           "displayLocationInDecimal": false,
    *           "encrypted": false,
    *           "externalId": false,
    *           "extraTypeInfo": null,
    *           "filterable": true,
    *           "filteredLookupInfo": null,
    *           "formulaTreatNullNumberAsZero": false,
    *           "groupable": true,
    *           "highScaleNumber": false,
    *           "htmlFormatted": false,
    *           "idLookup": true,
    *           "inlineHelpText": null,
    *           "label": "Lead ID",
    *           "length": 18,
    *           "mask": null,
    *           "maskType": null,
    *           "name": "Id",
    *           "nameField": false,
    *           "namePointing": false,
    *           "nillable": false,
    *           "permissionable": false,
    *           "picklistValues": [],
    *           "polymorphicForeignKey": false,
    *           "precision": 0,
    *           "queryByDistance": false,
    *           "referenceTargetField": null,
    *           "referenceTo": [],
    *           "relationshipName": null,
    *           "relationshipOrder": null,
    *           "restrictedDelete": false,
    *           "restrictedPicklist": false,
    *           "scale": 0,
    *           "searchPrefilterable": false,
    *           "soapType": "tns:ID",
    *           "sortable": true,
    *           "type": "id",
    *           "unique": false,
    *           "updateable": false,
    *           "writeRequiresMasterRead": false
    *         },
    *         {
    *           ....
    *         },
    *         .....
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function Describe(array $args) {
        $sobject = $args['sobject'];
        $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects/'. $sobject . '/describe';

        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $api = new RestAccess(array("Authorization" => $access_token));
        $result = $api->get($apiUrl);
        unset($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            $converted_result = $this->convertError($result, $sobject);
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        $converted_result = array(self::KEY_SUCCESS => true, self::KEY_RESULT => $response);
        return array(ApiConstants::KEY_RAW_RESPONSE => $converted_result);
    }

   /* GetContactInfo endpoint (Gets information about Contact or Lead)
    * @request POST api/v1/SalesForce/GetContactInfo
    * @parameters
    * Name                  | Placement       | Description
    * ----------------------|-----------------|----------------------------------------------------------
    * instanceUrl           | Body            | SalesForce API instance URL
    * phoneNumber           | Body            | Phone number to look for
    * sobjects              | Body            | Array of objects returned by ListObjects endpoint (Contact, Lead, CollaborationGroup, etc.)
    * searchPhoneTypes      | Body            | Array of "phone" type fields to be searched for phoneNumber
    * mandatoryFields       | Body            | Array of mandatory fields whose value need to be get
    * minNumOfMatchedDigits | Body            | Minimal number of digits to match the phone number
    * exactMatch            | Body            | Boolean: if true, exact match of complete phone number required
    * Authorization         | Headers         | 'Bearer ' . access_token
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "instanceUrl": "https://um1.salesforce.com",
    *   "phoneNumber": "+37491900046"
    *   "sobjects": [
    *     "Contact",
    *     "Lead"
    *   ],
    *   "searchPhoneTypes" {
    *     "Contact": [
    *        "Phone",
    *        "MobilePhone",
    *        "Fax",
    *        "HomePhone",
    *        "OtherPhone"
    *     ],
    *     "Lead": [
    *        "Phone",
    *        "MobilePhone"
    *     ]
    *   },
    *   "mandatoryFields": {
    *     "Contact": [
    *        "LastName",
    *        "Level__c",
    *        "Languages__c"
    *     ],
    *     "Lead": [
    *        "LastName",
    *        "Company"
    *     ]
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *        "Contact": [
    *          {
    *            "FirstName": "Armen",
    *            "LastName": "Movsisyan",
    *            "Email": armen@epygi.com,
    *            "Description": null,
    *            "MobilePhone": null,
    *            "Phone": "551606",
    *            "Fax": null,
    *            "id": "0033z00002hApHkAAK"
    *          },
    *          ...........
    *        }
    *      ]
    *      "Lead": [
    *         ......
    *      ]
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetContactInfo(array $args) {
        $sobjects = $args['sobjects'];
        $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/query';
        $phoneNumber = $args['phoneNumber'];
        $searchPhoneTypes = $args['searchPhoneTypes'];
        $mandatoryFields = $args['mandatoryFields'];
        $minNumOfMatchedDigits = $args['minNumOfMatchedDigits'];
        $exactMatch = $args['exactMatch'];
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];

        if (!count($sobjects)) {
            $response  = array (
                self::KEY_SUCCESS => true,
                self::KEY_RESULT => array ()
            );
            return $response;
        }
        
        foreach ($sobjects as $sobject) {
            if (!count($searchPhoneTypes[$sobject])) {
                $response  = array (
                    self::KEY_SUCCESS => false,
                    self::KEY_ERROR => array (
                        self::KEY_CODE => self::ARGUMENT_MAY_NOT_BE_EMPTY_ERR_CODE,
                        self::KEY_MESSAGE => sprintf($this->ARGUMENT_MAY_NOT_BE_EMPTY_ERR_MSG, 'searchPhoneTypes: ' .$sobject)
                    )
                );
        
                throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_SF_CALL_FAIL => $response), $response);
            }
        }

        if ($minNumOfMatchedDigits > strlen($phoneNumber)) {
            $response  = array (
                self::KEY_SUCCESS => true,
                self::KEY_RESULT => array ()
            );
            return $response;
        }

        $converted_result = array (
            ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true)
        );

        foreach ($sobjects as $sobject) {

            $str_where = array();
            $str_what = array("FirstName", "LastName", "Email", "Description");

            foreach ($searchPhoneTypes[$sobject] as $phonetype) {
                if ($exactMatch === true) {
                    $str_where[] = "$phonetype = '$phoneNumber'";
                } else {
                    $str_where[] = "$phonetype like '%$phoneNumber%'";
                }
                if (false === array_search ($phonetype, $str_what)) {
                    $str_what[] = $phonetype;
                }
            }
            $where2 = implode(" or ", $str_where);

            foreach ($mandatoryFields[$sobject] as $mfld) {
                if (false === array_search ($mfld, $str_what)) {
                    $str_what[] = $mfld;
                }
            }
            $what = implode(", ", $str_what);

            $where = "IsDeleted = false" . (($sobject == "Lead") ? " and IsConverted = false" : '');
            $query = "SELECT $what FROM $sobject where $where and ($where2)";
            $api = new RestAccess(array("Authorization" => $access_token));
            $result = $api->get($apiUrl, array("q" => $query));
            unset($api);
    
            if ($result[ApiConstants::KEY_ERROR]) {
                $converted_result = $this->convertError($result, $sobject);
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
            }
    
            $converted_result[ApiConstants::KEY_RAW_RESPONSE][self::KEY_RESULT][$sobject] = $this->convertContactInfo($result[ApiConstants::KEY_RESPONSE]);
        }

        return $converted_result;
    }

    /* GetResourceInfo endpoint
    * @request POST api/v1/SalesForce/GetResourceInfo
    * @parameters
    * Name                        | Placement    | Description
    * ----------------------------|--------------|---------------------------------------------------------------------
    * instanceUrl                 | Body         | SalesForce API instance URL
    * sobjects                    | Body         | Array of objects returned by ListObjects endpoint (Contact, Lead, CollaborationGroup, etc.)
    * Authorization               | Headers      | 'Bearer ' . access_token
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "instanceUrl": "https://um1.salesforce.com",
    *   "sobjects": [
    *     "Contact",
    *     "Lead"
    *   ]
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contact": {
    *           "success": true
    *       },
    *       "Lead": {
    *           "errorCode": "NOT_FOUND",
    *           "message": "The requested resource does not exist",
    *           "success": false
    *       }
    *   }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetResourceInfo(array $args) {
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $sobjects = $args['sobjects'];

        if (!count($sobjects)) {
            $response  = array (
                self::KEY_SUCCESS => false,
                self::KEY_ERROR => array (
                    self::KEY_CODE => self::ARGUMENT_MAY_NOT_BE_EMPTY_ERR_CODE,
                    self::KEY_MESSAGE => sprintf($this->ARGUMENT_MAY_NOT_BE_EMPTY_ERR_MSG, 'sobjects')
                )
            );

            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                array(self::KEY_SF_CALL_FAIL => $response), $response);
        }

        $info = array();
    
        $generic_success = false;
        foreach ($sobjects as $sobject) {
            $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects/'. $sobject . '/describe';
            $api = new RestAccess(array("Authorization" => $access_token));
            $result = $api->get($apiUrl);
            unset($api);

            if ($result[ApiConstants::KEY_ERROR] && $result[ApiConstants::KEY_ERROR_CODE] != 404) { //404- Not found
                $converted_result = $this->convertError($result, $sobject);
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
            }

            if ($result[ApiConstants::KEY_ERROR] || $result[ApiConstants::KEY_RESPONSE][0][self::KEY_ERROR_CODE]) {
                $info[$sobject] = $result[ApiConstants::KEY_RESPONSE][0];
                $info[$sobject][self::KEY_SUCCESS] = false;
            } else {
                $info[$sobject][self::KEY_CREATEABLE] = ($result[ApiConstants::KEY_RESPONSE][self::KEY_CREATEABLE]) ? true : false;
                $info[$sobject][self::KEY_DELETEABLE_RESP] = ($result[ApiConstants::KEY_RESPONSE][self::KEY_DELETEABLE_REQW]) ? true : false;
                $info[$sobject][self::KEY_UPDATEABLE] = ($result[ApiConstants::KEY_RESPONSE][self::KEY_UPDATEABLE]) ? true : false;
                $info[$sobject][self::KEY_SUCCESS] = true;
                $generic_success = true;
            }
        }
    
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (
                self::KEY_SUCCESS => $generic_success,
                self::KEY_RESULT => $info
            )
        );

        return $response;
    }

   /* GetMandatoryFields endpoint
    * @request POST api/v1/SalesForce/GetMandatoryFields
    * @parameters
    * Name                        | Placement    | Description
    * ----------------------------|--------------|---------------------------------------------------------------------
    * instanceUrl                 | Body         | SalesForce API instance URL
    * sobjects                    | Body         | Array of objects returned by ListObjects endpoint (Contact, Lead, CollaborationGroup, etc.)
    * userDefinedMandatoryFields  | Body         | Array of user defined mandatory fields
    * Authorization               | Headers      | 'Bearer ' . access_token
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "instanceUrl": "https://um1.salesforce.com",
    *   "sobjects": [
    *     "Contact",
    *     "Lead"
    *   ],
    *   "userDefinedMandatoryFields": {
    *     "Contact": [
    *       "Level__c",
    *       "Languages__c"
    *     ],
    *     "Lead": [
    *       "CleanStatus"
    *     ]
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contact": [
    *         {
    *           "name": "LastName",
    *           "label": "Last Name",
    *           "type": "string",
    *           "isunique": false,
    *           "default": null,
    *           "picklistValues": []
    *         },
    *         {
    *           "name": "Level__c",
    *           "label": "Level",
    *           "type": "picklist",
    *           "isunique": false,
    *           "default": null,
    *           "picklistValues": [
    *              {
    *                 "active": true,
    *                 "defaultValue": false,
    *                 "label": "Secondary",
    *                 "validFor": null,
    *                 "value": "Secondary"
    *              },
    *              {
    *                 "active": true,
    *                 "defaultValue": false,
    *                 "label": "Tertiary",
    *                 "validFor": null,
    *                 "value": "Tertiary"
    *              }
    *           ]
    *         },
    *         .....
    *       ],
    *       "Lead": [
    *         {
    *           "name": "LastName",
    *           "label": "Last Name",
    *           "type": "string",
    *           "isunique": false,
    *           "default": null,
    *           "picklistValues": []
    *         },
    *         .....
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetMandatoryFields(array $args) {
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $sobjects = $args['sobjects'];
    
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true)
        );
        
        if (!count($sobjects)) {
            $response[self::KEY_RESULT] = array();
            return $response;
        }

        foreach ($sobjects as $sobject) {
            $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects/'. $sobject . '/describe';
            $userDefinedMandatoryFields = ($args['userDefinedMandatoryFields'][$sobject] ? $args['userDefinedMandatoryFields'][$sobject] : array());
            $userDefinedMandatoryFields[] = 'OwnerId'; //explicitly set OwnerId
            
            $api = new RestAccess(array("Authorization" => $access_token));
            $result = $api->get($apiUrl);
            unset($api);

            if ($result[ApiConstants::KEY_ERROR]) {
                $converted_result = $this->convertError($result, $sobject);
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
            }

            $fields = $result[ApiConstants::KEY_RESPONSE][self::KEY_FIELDS];

            $mandatory_fields = array();
            foreach ($fields as $fld) {
                if (($fld['createable'] === true && $fld["defaultedOnCreate"] === false && $fld["nillable"] === false) || (in_array($fld["name"], $userDefinedMandatoryFields) === true)) {
                    //the field looks to be mandatory
                    $picklistValues = array();
                    if ($fld['picklistValues']) {
                        $picklistValues = $fld['picklistValues'];
                    }
                    foreach ($picklistValues as $key => $val) {
                        if ($val['active'] !== true) {
                            unset ($picklistValues[$key]);
                        }
                    }
                    $mandatory_fields[] = array (
                        'name' => $fld['name'],
                        'label' => $fld['label'],
                        'type' => $fld['type'],
                        'isunique' => $fld['unique'],
                        'default' => $fld['defaultValue'],
                        'picklistValues' => $picklistValues
                    );
                }
            }

            $response[ApiConstants::KEY_RAW_RESPONSE][self::KEY_RESULT][$sobject] = $mandatory_fields;
        }

        return $response;
    }

   /* GetPhonetypeFields endpoint
    * @request POST api/v1/SalesForce/GetPhonetypeFields
    * @parameters
    * Name           | Placement       | Description
    * ---------------|-----------------|----------------------------------------------------------------------
    * instanceUrl    | Body             | SalesForce API instance URL
    * sobjects       | Body             | Array of objects returned by ListObjects endpoint (Contact, Lead, etc.)
    * Authorization  | Headers          | 'Bearer ' . access_token
    * @sampleBody
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "instanceUrl": "https://um1.salesforce.com",
    *   "sobjects": [
    *     "Contact",
    *     "Lead"
    *   ],
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    * @sampleResponse
    * ~~~~~~~~~~~~~~~~~~~~~
    * {
    *   "rawResponse": {
    *     "success": true,
    *     "result": {
    *       "Contact": [
    *         {
    *           "name": "Phone",
    *           "label": "Business Phone",
    *           "editable": true
    *         },
    *         {
    *           "name": "Fax",
    *           "label": "Business Fax",
    *           "editable": true
    *         },
    *         {
    *           "name": "MobilePhone",
    *           "label": "Mobile Phone",
    *           "editable": true
    *         }
    *       ],
    *       "Lead": [
    *         {
    *           "name": "Phone",
    *           "label": "Phone",
    *           "editable": true
    *         }
    *       ]
    *     }
    *   }
    * }
    * OR
    * {
    *   "rawResponse": {
    *     "success": false,
    *     "error": {
    *       "code": "ERROR_CODE",
    *       "message": "Error Message"
    *     }
    *   }
    * }
    * ~~~~~~~~~~~~~~~~~~~~~
    */
    public function GetPhonetypeFields(array $args) {
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $sobjects = $args['sobjects'];
    
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true)
        );
    
        if (!count($sobjects)) {
            $response[self::KEY_RESULT] = array();
            return $response;
        }
        
        foreach ($sobjects as $sobject) {
            $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects/'. $sobject . '/describe';
            $api = new RestAccess(array("Authorization" => $access_token));
            $result = $api->get($apiUrl);
            unset($api);

            if ($result[ApiConstants::KEY_ERROR]) {
                $converted_result = $this->convertError($result, $sobject);
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
            }

            $fields = $result[ApiConstants::KEY_RESPONSE][self::KEY_FIELDS];

            $phonetype_fields = array();
            foreach ($fields as $fld) {
                if ($fld['type'] == 'phone') {
                    $phonetype_fields[] = array (
                        'name' => $fld['name'],
                        'label' => $fld['label'],
                        'editable' => $fld['updateable'] || $fld['createable'],
                    );
                }
            }

            $response[ApiConstants::KEY_RAW_RESPONSE][self::KEY_RESULT][$sobject] = $phonetype_fields;
        }

        return $response;
    }

    /* GetListOfFields endpoint
     * @request POST api/v1/SalesForce/GetListOfFields
     * @parameters
     * Name                        | Placement    | Description
     * ----------------------------|--------------|---------------------------------------------------------------------
     * instanceUrl                 | Body         | SalesForce API instance URL
     * sobjects                    | Body         | Array of objects returned by ListObjects endpoint (Contact, Lead, CollaborationGroup, etc.)
     * Authorization               | Headers      | 'Bearer ' . access_token
     * @sampleBody
     * ~~~~~~~~~~~~~~~~~~~~~
     * {
     *   "instanceUrl": "https://um1.salesforce.com",
     *   "sobjects": [
     *     "Contact",
     *     "Lead"
     *   ],
     * }
     * ~~~~~~~~~~~~~~~~~~~~~
     * @sampleResponse
     * ~~~~~~~~~~~~~~~~~~~~~
     * {
     *   "rawResponse": {
     *     "success": true,
     *     "result": [
     *       "Id",
     *       "IsDeleted",
     *       "MasterRecordId",
     *       "AccountId",
     *       "LastName",
     *       "FirstName",
     *       "Salutation",
     *       "Name"
     *       .....
     *     ]
     *   }
     * }
     * OR
     * {
     *   "rawResponse": {
     *     "success": false,
     *     "error": {
     *       "code": "ERROR_CODE",
     *       "message": "Error Message"
     *     }
     *   }
     * }
     * ~~~~~~~~~~~~~~~~~~~~~
     */
    public function GetListOfFields(array $args) {
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $sobjects = $args['sobjects'];
    
        $response = array (
            ApiConstants::KEY_RAW_RESPONSE => array (self::KEY_SUCCESS => true)
        );
    
        if (!count($sobjects)) {
            $response[self::KEY_RESULT] = array();
            return $response;
        }
        
        foreach ($sobjects as $sobject) {
            $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects/'. $sobject . '/describe';
            $api = new RestAccess(array("Authorization" => $access_token));
            $result = $api->get($apiUrl);
            unset($api);

            if ($result[ApiConstants::KEY_ERROR]) {
                $converted_result = $this->convertError($result, $sobject);
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
            }

            $fields = $result[ApiConstants::KEY_RESPONSE][self::KEY_FIELDS];
            $list_of_fields = array();
            foreach ($fields as $fld) {
                if (($fld['type'] == 'address') || ($fld['type'] == "reference") || ($fld['type'] == "id")) {
                    //skip for now 'address', 'reference' and 'id' types
                    continue;
                }
                $list_of_fields[] = $fld['name'];
            }

            $response[ApiConstants::KEY_RAW_RESPONSE][self::KEY_RESULT][$sobject] = $list_of_fields;
        }

        return $response;
    }

    /* CreateContact endpoint (Creates/Updates Contact or Lead)
     * @request POST api/v1/SalesForce/CreateContact
     * @parameters
     * Name                    | Placement | Description
     * ------------------------|-----------|-------------------------------------------------------------------------------
     * instanceUrl             | Body      | SalesForce API instance URL
     * loginUserName           | Body      | Login User name for client's SalesFoce CRM (this is default value for OwnerName)
     * recordType              | Body      | Record type: Contact or Lead
     * FirstName               | Body      | First Name
     * LasttName               | Body      | Last Name
     * MobilePhone             | Body      | Mobile Phone
     * Phone                   | Body      | Office Phone
     * HomePhone               | Body      | Home Phone
     * AssistantPhone          | Body      | Assistant Phone
     * OtherPhone              | Body      | Secondary Phone
     * Fax                     | Body      | Fax
     * Email                   | Body      | Email Address
     * Description             | Body      | Description
     * OwnerName               | Body      | Owner (default is logged in username)
     * OwnerId                 | Body      | OwnerId (default is logged in user's id)
     * customManadatoryFields  | Body      | List of name/value pairs
     * id                      | Body      | In case if we do update the contact - id of the contact to be updated
     * Authorization           | Headers   | 'Bearer ' . access_token
     * @sampleBody
     * ~~~~~~~~~~~~~~~~~~~~~
     * {
     *   "instanceUrl": "https://um1.salesforce.com",
     *   "loginUserName": "armen@epygi.com",
     *   "recordType": "Contact",
     *   "firstname": "Armen",
     *   "lastname": "Movsisyan",
     *   "mobile": "091998877",
     *   "homephone": "556677",
     *   "phone": "+181812345678",
     *   "assistantphone": 181812345678,
     *   "email": "armen@movsisyan.com",
     *   "description": "Put some description here",
     *   "customManadatoryFields": [
     *     {
     *       "name": "Level__c",
     *       "value": "Primary"
     *     },
     *     {
     *       "name": "Languages__c",
     *       "value": "Armenian, Russian, English"
     *     }
     *   ]
     * }
     * ~~~~~~~~~~~~~~~~~~~~~
     * @sampleResponse
     * ~~~~~~~~~~~~~~~~~~~~~
     * {
     *   "rawResponse": {
     *     "success": true,
     *     "result": [
     *       {
     *         "id": "0033z00002hBEOhAAO",
     *         "errors": []
     *       }
     *     ]
     *   }
     * }
     * OR
     * {
     *   "rawResponse": {
     *     "success": false,
     *     "error": {
     *       "code": "ERROR_CODE",
     *       "message": "Error Message"
     *     }
     *   }
     * }
     * ~~~~~~~~~~~~~~~~~~~~~
     */
    public function CreateContact(array $args) {
        $instanceUrl = $args['instanceUrl'];
        $loginUserName = trim($args['loginUserName']);
        $recordType = trim($args['recordType']);
        $ownerName = ($args['OwnerName']) ? trim($args['OwnerName']) : $loginUserName;
        $ownerId = trim($args['OwnerId']);
        $customManadatoryFields = $args['customManadatoryFields'];
        $id = $args['id'];
    
        $apiUrl = rtrim($args["instanceUrl"],'/') . "/services/data/" . self::API_VERSION . "/sobjects/$recordType";
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $headers = array ('Authorization' => $access_token, 'Content-Type' => 'application/json');
        $api = new RestAccess($headers);
    
        $element_array = array ();
        if ($args["FirstName"] != '') {
            $element_array["FirstName"] = trim($args['FirstName']);
        }
        if ($args["LastName"] != '') {
            $element_array["LastName"] = trim($args['LastName']);
        }
        if ($args["Email"] != '') {
            $element_array["Email"] = trim($args['Email']);
        }
        if ($args["Description"] != '') {
            $element_array["Description"] = trim($args['Description']);
        }
        if ($ownerId) {
            $element_array["OwnerId"] = $ownerId;
        }
        if ($args["MobilePhone"] != '') {
            $element_array["MobilePhone"] = trim($args['MobilePhone']);
        }
        if ($args["Phone"] != '') {
            $element_array["Phone"] = trim($args['Phone']);
        }
        if ($args["HomePhone"] != '') {
            $element_array["HomePhone"] = trim($args['HomePhone']);
        }
        if ($args["AssistantPhone"] != '') {
            $element_array["AssistantPhone"] = trim($args['AssistantPhone']);
        }
        if ($args["OtherPhone"] != '') {
            $element_array["OtherPhone"] = trim($args['OtherPhone']);
        }
        if ($args["Fax"] != '') {
            $element_array["Fax"] = trim($args['Fax']);
        }

        foreach ($customManadatoryFields as $fld) {
            if ($element_array[$fld["name"]]) {
                continue;
            }
            $element_array[$fld["name"]] = $fld["value"];
        }

        if (!$element_array["OwnerId"] && $ownerName) {
            $this->getOwnerID($instanceUrl, $ownerName);
            $element_array["OwnerId"] = $this->ownerId;
        }
        
        if ($id) {
            $apiUrl .= "/$id";
            $result = $api->patch($apiUrl, $element_array);
        } else {
            $result = $api->post($apiUrl, $element_array);
        }
        unset ($api);

        if ($result[ApiConstants::KEY_ERROR]) {
            $converted_result = $this->convertError($result);
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
        }
    
        $success = (isset($result[ApiConstants::KEY_RESPONSE]["success"]) ? $result[ApiConstants::KEY_RESPONSE]["success"] : (($result[ApiConstants::KEY_ERROR_CODE]) ? false : true));
    
        unset($result[ApiConstants::KEY_RESPONSE]["success"]);
        $response = array (
            self::KEY_SUCCESS => $success,
            self::KEY_RESULT => $result[ApiConstants::KEY_RESPONSE]
        );
        
        if (!$success) {
            $err = reset($result[ApiConstants::KEY_RESPONSE]['errors']);
            throw new EndpointException($err, ApiConstants::http_Bad_Request,
                array(self::KEY_SF_CALL_FAIL => $response), $response);
        }

        return array(ApiConstants::KEY_RAW_RESPONSE => $response);
    }

    /* PutCDR endpoint (Export CDR in SalesForce)
     * @request POST api/v1/SalesForce/PutCDR
     * @parameters
     * Name                  | Placement       | Description
     * ----------------------|-----------------|----------------------------------------------------------
     * instanceUrl           | Body            | SalesForce API instance URL
     * phoneNumber           | Body            | Phone number to look for in Contacts and Leads
     * searchPhoneTypes      | Body            | Array of "phone" type fields to be searched for phoneNumber
     * ActivityDateTime      | Body            | CDR Date & Time
     * Subject               | Body            | Subject
     * DurationInMinutes     | Body            | Duration in minutes
     * Description           | Body            | Description
     * Authorization         | Headers         | 'Bearer ' . access_token
     * @sampleBody
     * ~~~~~~~~~~~~~~~~~~~~~
     * {
     *   "instanceUrl": "https://um1.salesforce.com",
     *   "phoneNumber": "+37491900046"
     *   "searchPhoneTypes" {
     *     "Contact": [
     *        "Phone",
     *        "MobilePhone",
     *        "Fax",
     *        "HomePhone",
     *        "OtherPhone"
     *     ],
     *     "Lead": [
     *        "Phone",
     *        "MobilePhone"
     *     ]
     *   },
     *   "ActivityDateTime": "2020-08-19T12:55:35Z"
     *   "Subject": "QX50 Inbound Call from 18182315431 to 11456",
     *   "DurationInMinutes": "5",
     *   "Description": "call forwarded"
     * }
     * ~~~~~~~~~~~~~~~~~~~~~
     * @sampleResponse
     * ~~~~~~~~~~~~~~~~~~~~~
     * {
     *   "rawResponse": {
     *     "success": true,
     *     "result": [
     *       {
     *         "id": "0033z00002hBEOhAAO",
     *         "success": true,
     *         "errors": []
     *       }
     *     ]
     *   }
     * }
     * OR
     * {
     *   "rawResponse": {
     *     "success": false,
     *     "error": {
     *       "code": "ERROR_CODE",
     *       "message": "Error Message"
     *     }
     *   }
     * }
     * ~~~~~~~~~~~~~~~~~~~~~
     */
    public function PutCDR(array $args) {
        $phoneNumber = $args['phoneNumber'];
        $searchPhoneTypes = $args['searchPhoneTypes'];
        $ActivityDateTime = $args['ActivityDateTime'];
        $Subject = $args['Subject'];
        $DurationInMinutes = $args['DurationInMinutes'];
        $Description = $args['Description'];
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];

        $whoId = array();
        $sobjects = array ("Contact", "Lead");
        $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/query';
        if ($phoneNumber) {
            foreach ($sobjects as $sobject) {
                if (count($searchPhoneTypes[$sobject])) {
                    $str_where = array();
                    foreach ($searchPhoneTypes[$sobject] as $phonetype) {
                        $str_where[] = "$phonetype = '$phoneNumber'";
                    }
                    $where = "IsDeleted = false" . (($sobject == "Lead") ? " and IsConverted = false" : '');
                    $where2 = implode(" or ", $str_where);
                    $query = "SELECT Id FROM $sobject where $where and ($where2)";
                    $api = new RestAccess(array("Authorization" => $access_token));
                    $result = $api->get($apiUrl, array("q" => $query));
                    unset($api);

                    if (!$result[ApiConstants::KEY_ERROR]) {
                        foreach ($result[ApiConstants::KEY_RESPONSE][self::KEY_RECORDS] as $record) {
                            $whoId[] = $record[self::KEY_ID];
                        }
                    }
                }
            }
        }

        $apiUrl = rtrim($args['instanceUrl'],'/') . '/services/data/' . self::API_VERSION . '/sobjects/Event';
        $headers = array ('Authorization' => $access_token, 'Content-Type' => 'application/json');
        $results = array();
        $success = 1;
        if (count($whoId)) {
            foreach($whoId as $id) {
                $api = new RestAccess($headers);
                $result = $api->post($apiUrl, array("Subject" => $Subject, "DurationInMinutes" => $DurationInMinutes, "ActivityDateTime" => $ActivityDateTime, "WhoId" => $id, "Description" => $Description));
                unset ($api);

                if ($result[ApiConstants::KEY_ERROR]) {
                    $converted_result = $this->convertError($result);
                    throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                        array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
                }

                $success &= $result[ApiConstants::KEY_RESPONSE][self::KEY_SUCCESS];
                $results[] = $result[ApiConstants::KEY_RESPONSE];
            }
        } else {
            $api = new RestAccess($headers);
            $result = $api->post($apiUrl, array("Subject" => $Subject, "DurationInMinutes" => $DurationInMinutes, "ActivityDateTime" => $ActivityDateTime, "Description" => $Description));
            unset ($api);

            if ($result[ApiConstants::KEY_ERROR]) {
                $converted_result = $this->convertError($result);
                throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                    array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
            }

            $success = $result[ApiConstants::KEY_RESPONSE][self::KEY_SUCCESS];
            $results[] = $result[ApiConstants::KEY_RESPONSE];
        }

        $response = array (
            self::KEY_SUCCESS => boolval($success),
            self::KEY_RESULT => $results
        );

        return array(ApiConstants::KEY_RAW_RESPONSE => $response);
    }

    private function getOwnerID($instanceUrl, $ownerName) {
        $apiUrl = rtrim($instanceUrl,'/') . '/services/data/' . self::API_VERSION . '/query';
        $access_token = $_SERVER['HTTP_AUTHORIZATION'];
        $api = new RestAccess(array("Authorization" => $access_token));
        $query = "SELECT Id FROM User where Username='$ownerName'";
        $result = $api->get($apiUrl, array("q" => $query));
        unset ($api);
    
        if ($result[ApiConstants::KEY_ERROR]) {
            $converted_result = $this->convertError($result);
            throw new EndpointException($result[ApiConstants::KEY_ERROR_MESSAGE], $result[ApiConstants::KEY_ERROR_CODE],
                array(self::KEY_SF_CALL_FAIL => $converted_result), $converted_result);
        }

        $response = $result[ApiConstants::KEY_RESPONSE];

        $this->ownerId = $result[ApiConstants::KEY_RESPONSE][self::KEY_RECORDS][0][self::KEY_ID];

        if (!$this->ownerId) {
            $response  = array (
                self::KEY_SUCCESS => false,
                self::KEY_ERROR => array (
                    self::KEY_CODE => self::OWNER_ID_NOT_FOUND_ERR_CODE,
                    self::KEY_MESSAGE => sprintf($this->OWNER_ID_NOT_FOUND_ERR_MSG, $ownerName)
                )
            );

            throw new EndpointException($response[self::KEY_ERROR][self::KEY_MESSAGE], ApiConstants::http_Bad_Request,
                    array(self::KEY_SF_CALL_FAIL => $response), $response);
        }

        return $response;
    }

    private function convertError($result, $sobject = '') {
        if ($result[ApiConstants::KEY_RESPONSE][self::KEY_ERROR]) {
            $error = $result[ApiConstants::KEY_RESPONSE][self::KEY_ERROR];
        } elseif($result[ApiConstants::KEY_RESPONSE][0][self::KEY_ERROR_CODE]) {
            $error = $result[ApiConstants::KEY_RESPONSE][0][self::KEY_ERROR_CODE];
        } else {
            $error = $result[ApiConstants::KEY_ERROR_CODE];
        }
    
        if ($result[ApiConstants::KEY_RESPONSE][self::KEY_ERROR_DESCRIPTION]) {
            $error_description = (($sobject) ? $sobject . ": " : "") . $result[ApiConstants::KEY_RESPONSE][self::KEY_ERROR_DESCRIPTION];
        } elseif($result[ApiConstants::KEY_RESPONSE][0][self::KEY_MESSAGE]) {
            $error_description = (($sobject) ? $sobject . ": " : "") . $result[ApiConstants::KEY_RESPONSE][0][self::KEY_MESSAGE];
        } else {
            $error_description = (($sobject) ? $sobject . ": " : "") . $result[ApiConstants::KEY_ERROR_MESSAGE];
        }
        
        return array(
            self::KEY_SUCCESS => false,
            self::KEY_ERROR => array(
                self::KEY_CODE => $error,
                self::KEY_MESSAGE => $error_description
            )
        );
    }

    private function convertContactInfo($response) {
        $records = $response[self::KEY_RECORDS];
        foreach ($records as $key => $rec) {
            $record_url = $rec["attributes"]["url"];
            $id = $this->getRecordIdFromURL($record_url);
            unset($records[$key]['attributes']);
            $records[$key][self::KEY_ID] = $id;
        }
        return $records;
    }

    private function getRecordIdFromURL($url, $module) {
        $id_position = strrpos($url, "/");
        if ($id_position === false) {
            return null;
        }
        return substr($url,$id_position+1);
    }
}
?>
