<?php
namespace Epygi;

use Epygi\Enum\Role as ERole;
use Exception;
use Mvc;

class Auth
{
    const AUTH_LOGIN_URL = '/login';
    const PENDING_EVENTS_FILE = '/var/state/new_pending_events.db';
    const SESSION_DURATION = 900; // 900 seconds = 15 minutes (60 * 15)
    const SESSION_PREFIX = 'EPYGI_AUTH';
    const AUTH_2FA_PASS_GEN_EXEC_PATH = '/bin/hotp';

    /**
     * Returns the CGI session id stored by the login form.  We need this ID to pass to our CGI wrapper in order to
     * be properly authenticated by the FastCGI pages.
     * @return string - cgi session id
     */
    public static function getCgiSessionId()
    {
        return self::session()->get('cgi_session_id');
    }

    /**
     * Returns a pretty name to display based on the username and role of the logged in user.
     * @return string - pretty name
     */
    public static function getPrettyName()
    {
        $role = self::getRole();
        $username = self::getUsername();

        switch ($role) {
            case ERole::ADMIN:
            case ERole::LOCALADMIN:
                return sprintf(_('Administrator (%s)'), $username);

            case ERole::EXTENSION:
                return sprintf(_('Extension (%d)'), $username);

            case ERole::CONFERENCE:
                return sprintf(_('Conference (%d)'), $username);
		
    	    case ERole::RECORDING:
        	return sprintf(_('Recording (%d)'), $username);		
	
            case ERole::SUPERADMIN:
                return sprintf(_('SuperAdministrator (%s)'), $username);

            case ERole::VOICEMAILBOX:
                return sprintf(_('Voice MailBox (%d)'), $username);
	}

        // WTF?  This shouldn't happen!
        return _('Unknown User');
    }

    /**
     * Returns the role of the user.
     * @return string - user role (admin, localadmin, extension, conference)
     */
    public static function getRole()
    {
        return self::session()->role;
    }

    /**
     * Fetches the username from session storage.
     * @return string - username (admin, localadmin, 123, etc)
     */
    public static function getUserRole()
    {
        return self::session()->userrole;
    }

    /**
     * Fetches the username from session storage.
     * @return string - username (admin, localadmin, 123, etc)
     */
    public static function getUsername()
    {
        return self::session()->username;
    }

    public static function hasPendingEvents()
    {
        // only show this if the user is properly authorized (admins only)
        switch (self::getRole()) {
            case ERole::ADMIN:
            case ERole::LOCALADMIN:
	    {
	    if ( file_exists(self::PENDING_EVENTS_FILE) ) {
		$s=@stat(self::PENDING_EVENTS_FILE);
		if( !$s ) {
		    return false; //Couldnt stat file
		}
		//if character special file
		if (  decoct($s['mode'] & 0020000) ) {
		    $cmd="/bin/cat " .self::PENDING_EVENTS_FILE;
		    $content=shell_exec($cmd);
		    if( strlen($content) > 0 ) {
			return true;
		    }
		    else {
			return false;
		    }
		}
		return filesize(self::PENDING_EVENTS_FILE);
		}
	    }
        }

        // not authorized
        return false;
    }

    public static function isAdmin()
    {
        return self::isRole(ERole::ADMIN);
    }

    /**
     * If we have a username set to a non-empty value, we are logged in already.
     * @return bool - true if username is set, else false
     */
    public static function isAuth()
    {
        return (boolean) self::session()->username;
    }

    public static function isConference()
    {
        return self::isRole(ERole::CONFERENCE);
    }

    public static function isExtension()
    {
        return self::isRole(ERole::EXTENSION);
    }

    public static function isRecording()
    {
	return self::isRole(ERole::RECORDING);
    }

    public static function isLocalAdmin()
    {
        return self::isRole(ERole::LOCALADMIN);
    }

    public static function isSuperAdmin()
    {
        return self::isRole(ERole::SUPERADMIN);
    }

    public static function isVoiceMailBox()
    {
        return self::isRole(ERole::VOICEMAILBOX);
    }

    /**
     * Accepts a list of roles.  If the user is any one of the matched roles, then return true.
     *
     * @param $role - role to check
     *
     * @return bool - true if user is one of the roles, false otherwise
     */
    public static function isRole($role)
    {
        // fetch list of roles to check
        $args = func_get_args();

        // is our role in the list?
        return in_array(self::getRole(), $args);
    }

    /**
     * Two Factor Authentication functions
     *
     */

    public static function is2faLoginEnabled($username)
    {
    	$config = Storage\ConfigFactory::acldb();
	if(isset($config[$username])) {
		$usercfg = $config[$username];
		$enabled = $usercfg->getBool('enabled', false);
		if( $enabled == true ){
			$enabled = $usercfg->getBool('enabled2fa', false);
			return $enabled;
		}
	}
	return false;
    }

    public static function is2faLoginRequired($username)
    {
	return self::is2faLoginEnabled($username);
    }

    public static function GetKeyForOneTimePassword($user)
    {
	$key='';
	try {
            $key = Shell\Command\CheckSAM::samGet2faKey($user);
        }
        catch(\Exception $e) {
            $key = '';
        }
	return $key;
    }
    
    public static function GetOneTimePassword($user)
    {
	$oneTimePassword = '';
	$key = self::GetKeyForOneTimePassword($user);

	$cmd = new Shell\Command(self::AUTH_2FA_PASS_GEN_EXEC_PATH);
	$cmd->opt('-b', $key );
	try {
		if ($cmd->execute() == 0) {
			$oneTimePassword = $cmd->data[0];
		}
	}
	catch(\Exception $e) {
		return '';
	}
	return $oneTimePassword;
    }

    public static function before2FalogIn($username, $role)
    {
        // clear session
        self::logOut(true);

        $session = self::session();
        self::sessionExpireRefresh();

        // save the username and role in temp session for 2FA
        $session->username2fa = $username;
        $session->role2fa = $role;
    }

    public static function Get2faLoginUser()
    {
	$session = self::session();
	return $session->username2fa;
    }

    public static function Get2faLoginUserRole()
    {
	$session = self::session();
	return $session->role2fa;
    }
	
    /**
     * Assign the currently logged in username/role to the session storage object for permanent session storage.
     *
     * @param $username
     * @param $role
     *
     * @return string - username
     * @throws Exception
     */
    public static function logIn($username, $role)
    {
        // clear session
        self::logOut(true);

        // save the role
        switch ($role) {
            case ERole::ADMIN:
                $username = 'admin'; // hard code username
                $userrole = '00000002'; // compatibility with CGIs
                break;

            case ERole::LOCALADMIN:
                $username = 'localadmin'; // hard code username
                $userrole = '00000004'; // compatibility with CGIs
                break;

            case ERole::EXTENSION:
                // make sure the "username" is a valid extension (digits only)
                $username = preg_replace('/\D/', '', $username);
                $userrole = '00000010'; // compatibility with CGIs
                break;
            case ERole::CONFERENCE:
                // make sure the "username" is a valid extension (digits only)
                $username = preg_replace('/\D/', '', $username);
                $userrole = '00000008'; // compatibility with CGIs
                break;

            case ERole::RECORDING:
                // make sure the "username" is a valid extension (digits only)
                $username = preg_replace('/\D/', '', $username);
                $userrole = '00000040'; // compatibility with CGIs
                break;
            case ERole::SUPERADMIN:
                $username = 'superadmin'; // hard code username
                $userrole = '00000080'; // compatibility with CGIs
                break;
            case ERole::VOICEMAILBOX:
                $username = preg_replace('/\D/', '', $username);
                $userrole = '00000100'; // compatibility with CGIs
                break;
            default:
                // sanity check (should never happen)
                throw new Exception('Invalid user role: ' . $role);
        }

        // fetch session object and reset our expiration
        $session = self::session();
        self::sessionExpireRefresh();

        // save the username and role in our session
        $session->username = $username;
        $session->role = $role;
        $session->userrole = $userrole;

        // Save new session on sessiondatadb
        self::setCgiSessionId(time());

        Storage\SessionData::setSession(
            self::getCgiSessionId(),
            self::getUsername(),
            self::getUserRole(),
            time() + self::SESSION_DURATION
        );
        
        // remove expired sessions from sessiondatadb
        Storage\SessionData::removeExpiredSessions();

        // we should be logged in now (sanity check)
        return self::isAuth();
    }

    /**
     * Log the user out and clear all auth session data.  We do not perform any redirects here because that is
     * handled by another function like rejectUnlessAuth().
     */
    public static function logOut($remember_want = false)
    {
        // maybe we already have a page to go to
        $want_uri = self::session()->want_uri;

        // remove from cgi session
        Storage\SessionData::removeSession(self::session()->cgi_session_id);

        // clear session storage
        self::session()->destroy();

        // keep track of the page we originally wanted
        if ($remember_want) {
            self::session()->want_uri = $want_uri;
        }
    }

    /**
     * If we are not logged in properly, log out all the way and redirect to the login screen.
     *
     * @param bool $capture_url - remember the url we wanted so we can redirect there after log in
     *
     * @return bool - true if logged in, redirect if not
     */
    public static function rejectUnlessAuth($capture_url = true)
    {
        // we are logged in
        if (self::isAuth()) {
            return true;
        }

        // make sure we are entirely logged out
        self::logOut(true);

        // save the page this user was originally going to
        if ($capture_url) {
            self::wantUriCapture();
        }

        // redirect
        header('Location: ' . self::AUTH_LOGIN_URL);
        exit;
    }

    /**
     * Return an initialized session storage object.  The object is responsible for making sure the PHP session has
     * been started and that the session data is loaded and ready to be used.  We also check the session expiration
     * time here and if the session has expired, all session data is wiped clean.  If the session has NOT expired,
     * the session is refreshed and a new expiration time is calculated.
     */
    private static function session()
    {
        static $session = null;

        // start a new session if necessary
        if (!$session) {
            $session = new Mvc\Session(self::SESSION_PREFIX);
            self::sessionExpireRefresh();
        }

        // return our session object
        return $session;
    }

    public static function sessionExpireRefresh()
    {
        $session = self::session();
/*	
		// session expired, clear the session (this will cause the user to be logged OUT!)
		if ($session->expire_timestamp < time()) {
			self::logOut(true);
		}
*/
        // Update existing session on sessiondatadb
        Storage\SessionData::updateSessionTime($session->cgi_session_id, time() + (self::SESSION_DURATION));

        // set a new expiration timestamp (even if we cleared above ... needed by logIn)
        $session->expire_timestamp = time() + (self::SESSION_DURATION);
    }

    /**
     * @param $cgi_session_id
     */
    public static function setCgiSessionId($cgi_session_id)
    {
        self::session()->cgi_session_id = $cgi_session_id;
    }

    public static function wantUriCapture()
    {
        // TODO: use Router to fetch current URI and want path
        self::session()->want_uri = Mvc\Param::server('REQUEST_URI');
    }

    /**
     * TODO: validate the want URI against valid Router paths.  You should be able to want something that doesn't
     * exist.
     */
    public static function wantUriRedirect()
    {
        // do we have a page to go to?
        $want_uri = trim(self::session()->want_uri);

        // we never *really* wanted these ... TODO: make this better
        $want_uri = preg_replace('#^/?#', '', $want_uri);
        $want_uri = preg_replace('#^/login#', '', $want_uri);
        $want_uri = preg_replace('#^/logout#', '', $want_uri);
        $want_uri = preg_replace('#^/index/login#', '', $want_uri);
        $want_uri = preg_replace('#^/index/logout#', '', $want_uri);

        // we have a page ... let's go there
        if ($want_uri) {
            // remove the page so we don't end up in infinite loop redirecting
            self::session()->want_uri = false;

            // redirect to the wanted page
            header('Location: ' . $want_uri);
            exit;
        }
    }
}