<?php
namespace Epygi\Form;

/**
 * Common validation functions used with forms.
 * TODO: find an efficient way to load validation functions from a plugin architecture so we don't have to load these
 * all the time ... perhaps use __invoke or __call.
 */
abstract class AbstractMultistepForm extends Engine
{
    protected function check_alpha_numeric($field)
    {
        $value = trim($this->getValue($field));
        if (!ctype_alnum($value)) {
            return _('No special characters allowed.');
        }

        // no error
        return false;
    }

    protected function check_dns_duplicate($field, $nameserver)
    {
        $value = trim($this->getValue($field));
        if ($value && $value == $nameserver) {
            return _('DNS Servers cannot be duplicate.');
        }

        //no duplicate
        return false;
    }

    protected function check_email($field)
    {
        $email = trim(strtolower($this->getValue($field)));

        // required should have already caught this
        if (!$email) {
            return false;
        }

        // not valid email
        if (!$this->validateEmail($email)) {
            return _(
                'The email address you entered does not appear to be valid.  Please correct your entry and try again.'
            );
        }

        // no error
        return false;
    }

    protected function check_ipv4($field)
    {
        $value = trim($this->getValue($field));

        // not a valid IP4v value
        if (!filter_var($value, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return _('Please enter a valid ip address.');
        }

        // no error
        return false;
    }

    protected function check_length($field, $maxlen, $minlen = 0)
    {
        // clean the field

        $this->$field = trim($this->$field);
        $value = $this->getValue($field);
        $len = sprintf('%d', strlen($value));

        // check the max len
        if ($len > $maxlen) {
            $err = _('The maximum length for this field is %d characters.  You have used %d characters.');
            return sprintf($err, $maxlen, $len);
        }

        // check the min len
        if ($minlen && $len < $minlen) {
            $err = _('The minimum length for this field is %d characters.  You have used %d characters.');
            return sprintf($err, $minlen, $len);
        }

        // no error
        return false;
    }

    /**
     * Validate a mac address
     *
     * @param $field
     *
     * @return string
     */
    protected function check_mac_address($field)
    {
        $value = trim($this->getValue($field));
        if (!preg_match("/^[0-9a-fA-F]{2}(?=([-:;.]?))(?:\\1[0-9a-fA-F]{2}){5}$/i", $value)) {
            return _('Please enter a valid mac address. ');
        }
    }

    protected function check_numeric($field)
    {
        // remove whitespace
        $this->$field = trim($this->$field);

        // strip commas (be nice)
        $this->$field = preg_replace('/,/', '', $this->$field);

        // remove all characters but digits
        if (!is_numeric($this->$field)) {
            return _('Please enter numbers only.');
        }

        // check that it is an integer
        $num = (int) $this->$field;
        if ($num != $this->$field) {
            return _('Please enter whole numbers only.');
        }

        // no error
        $this->$field = $num;
        return false;
    }

    protected function check_range($field, $min, $max)
    {
        // value must be numeric
        $err = $this->check_numeric($field);
        if ($err) {
            return $err;
        }

        // data is NOT in range min to max
        if ($this->$field > $max || $this->$field < $min) {
            return sprintf(_('The value you entered is not in the proper range %d-%d.'), $min, $max);
        }

        // no error
        return false;
    }

    protected function check_selection($field, $opts)
    {
        $this->$field = trim($this->$field);

        // found the value in our option keys
        if (isset($opts[$this->$field])) {
            return false;
        }

        // did not find the value
        return 'Invalid selection.';
    }

    function check_subnet_mask($field)
    {
        $mask = trim($this->getValue($field));

        $bin = decbin(ip2long($mask));
        if (!(strlen($bin) == 32 && !preg_match('/01/', $bin))) {
            return _('Please enter a valid netmask.');
        }

        return false;
    }

    function check_subnet_mask_bit_length($field, $max)
    {
        $mask = trim($this->getValue($field));
        $bin = decbin(ip2long($mask));
        $bit = strpos($bin, '0');
        if ($bit >= $max || $bit === false) {
            return _('Network mask is too large');
        }
        return false;
    }

    protected function numeric($field)
    {
        $value = $this->getValue($field);
        if (!is_numeric($value)) {
            return _('Must be a numeric value');
        }

        //no error
        return false;
    }

    //protected function isValidDate($year, $month, $day)
    //{
    //    // convert to timestamp
    //    $ts = mktime(12, 0, 0, $month, $day, $year);
    //
    //    // convert back
    //    $nyear = date('Y', $ts);
    //    $nmonth = date('m', $ts);
    //    $nday = date('d', $ts);
    //
    //    // compare
    //    if ($nyear != $year || $nmonth != $month || $nday != $day) {
    //        return false;
    //    }
    //
    //    // date looks fine
    //    return true;
    //}

    /**
     * http://stackoverflow.com/questions/1755144/how-to-validate-domain-name-in-php
     */
    public static function validateDomainName($domain_name)
    {
        return (preg_match('/^([a-z\d](-*[a-z\d])*)(\.([a-z\d](-*[a-z\d])*))*$/i', $domain_name) //valid chars check
            && preg_match('/^.{1,253}$/', $domain_name) //overall length check
            && preg_match('/^[^\.]{1,63}(\.[^\.]{1,63})*$/', $domain_name)); //length of each label
    }

    public static function validateEmail($value)
    {
        // not a valid IP4v value
        if (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
            return false;
        }

        // no error
        return true;
        //$pattern = '/^[-_a-z0-9]+(\.[-_a-z0-9]+)*@[-a-z0-9]+(\.[-a-z0-9]+)*\.[a-z]{2,6}$/';
        //return preg_match($pattern, $email);
    }

    /**
     * https://github.com/zendframework/zf2/blob/master/library/Zend/Validator/Ip.php
     * Validates an IPv4 address
     *
     * @param string $value
     *
     * @return bool
     */
    public static function validateIPv4($value)
    {
        if (preg_match('/^([01]{8}.){3}[01]{8}$/i', $value)) {
            // binary format  00000000.00000000.00000000.00000000
            $value = bindec(substr($value, 0, 8)) . '.' . bindec(substr($value, 9, 8)) . '.'
                . bindec(substr($value, 18, 8)) . '.' . bindec(substr($value, 27, 8));
        }
        elseif (preg_match('/^([0-9]{3}.){3}[0-9]{3}$/i', $value)) {
            // octet format 777.777.777.777
            $value = (int) substr($value, 0, 3) . '.' . (int) substr($value, 4, 3) . '.'
                . (int) substr($value, 8, 3) . '.' . (int) substr($value, 12, 3);
        }
        elseif (preg_match('/^([0-9a-f]{2}.){3}[0-9a-f]{2}$/i', $value)) {
            // hex format ff.ff.ff.ff
            $value = hexdec(substr($value, 0, 2)) . '.' . hexdec(substr($value, 3, 2)) . '.'
                . hexdec(substr($value, 6, 2)) . '.' . hexdec(substr($value, 9, 2));
        }

        $ip2long = ip2long($value);
        if ($ip2long === false) {
            return false;
        }

        return ($value == long2ip($ip2long));
    }
}