<?php
namespace Epygi\Grid;

abstract class Engine
{
    // defaults
    protected $default_order_field = '';
    protected $default_order_direction = 0;

    //grid now uses toolbar
    public $toolbar;

    // filter, order, action, pager FOAP!
    public $filter = null;
    public $order = null;
    public $action = null;
    public $pager = null;

    // records
    private $record_count = 0;
    private $records = false;

    /**
     * FOAP objects created here.  They will be initialized from the child init() method later.
     */
    public function __construct()
    {
        // FOAP!  we will configure and initialize these later
        $this->filter = new Filter();
        $this->order = new Order();
        $this->action = new Action();
        $this->pager = new Pager();
    }

    /**
     * @return int - number of records loaded
     */
    protected function getRecordCount()
    {
        return count($this->records);
    }

    /**
     * Return loaded, filtered, ordered, and paged data.
     * @return array - data
     */
    public function getRecords()
    {
        return is_array($this->records) ? $this->records : array();
    }

    public abstract function init();

    /**
     * We are searching for something, but nothing was found.
     * @return bool
     */
    public function isEmptySearch()
    {
        return $this->filter->hasValue() && $this->isNoResults();
    }

    /**
     * An error occurred loading the data.
     * @return bool
     */
    public function isError()
    {
        return !is_array($this->records);
    }

    /**
     * We successfully loaded the data, but there are no records to display.  No error, just no data.
     * @return bool
     */
    public function isNoResults()
    {
        return count($this->records) < 1;
    }

    /**
     * Inheriting class will define this method to load the records for display.
     * @return array - data
     */
    public abstract function loadRecords();

    /**
     * Call our child class to initialize FOAP objects.  Then, process any triggered actions.  Next, we load records
     * and sort/narrow the results by the filter, order, and pager objects.  This function will call the subclass
     * "loadRecords()" method to load pre-filtered records.
     */
    public function process()
    {
        // initialize FOAP objects
        $this->init();

        // process action callbacks
        $this->action->process($this);

        // load records
        $this->records = $this->loadRecords();

        // filter: init field/value and run data through our filter
        $this->filter->init();
        $this->records = $this->filter->filter($this->records);

        // order: initialize order defaults and sort our records
        $this->order->init($this->default_order_field, $this->default_order_direction);
        $this->order->order($this->records);

        // pager: set record count, and limit number of records
        $this->pager->init($this->getRecordCount());
        $this->records = $this->pager->limitOffset($this->records);
    }
}