<?php
namespace Epygi\Shell;

use Epygi;
use Exception;

class Command
{
    // debugging
    const DEBUG = true;
    private static $benchmark = false;
    private static $verbose = false;

    // options (command line arguments)
    public $data = false;

    // output
    public $result = false;
    private $opts = array();

    /**
     * The CLI binary goes on the OPTS stack first without any escaping.
     */
    public function __construct($command)
    {
        array_push($this->opts, $command);
    }

    public function __toString()
    {
        return join(' ', $this->opts);
    }

    /**
     * Render double right arrows to send STDOUT to append to a file.
     *
     * @param $file_name
     */
    public function append($file_name)
    {
        // add "double arrow" marks for command-line append
        array_push($this->opts, '>>');

        // escape file name
        $this->arg($file_name);
    }

    /**
     * Add an argument (with proper escaping) to the command line parameters.
     *
     * @param null $arg
     * @param null $default
     */
    public function arg($arg = null, $default = null)
    {
        // if arg is null (nothing passed in) nothing to do
        if (is_null($arg)) {
            return;
        }

        // the arg is empty, but we have a default alternative, use the default instead
        if (!is_null($default) && empty($arg)) {
            $arg = $default;
        }

        // escape shell argument
        $arg = escapeshellarg($arg);

        // bug fix: handle empty strings by replacing with empty quote
        if (strlen($arg) < 1) {
            $arg = "''";
        }

        // add argument to the command line
        array_push($this->opts, $arg);
    }

    /**
     * Useful for redirecting output of a command to a file.
     */
    public function arrow($with_errors = false)
    {
        array_push($this->opts, '>' . ($with_errors ? '&' : ''));
    }

    private function benchmark($start = 0, $end = 0)
    {
        // benchmarking not turned on
        if (!self::$benchmark) {
            return false;
        }

        // how many seconds elapsed from the start time?
        $elapsed = abs($end - $start);

        // log the message
        $this->debugLog(sprintf("[ELAPSED TIME: %.4f seconds]\n", $elapsed));
    }

    /**
     * Log messages to our error log only if debugging is turned on.
     */
    public function debugLog($message)
    {
        if (static::DEBUG) {
            error_log($message);
        }

        if (self::$verbose) {
            print $message . "<br/>\n";
        }
    }

    public static function enableBenchmark($enable = true)
    {
        self::$benchmark = (boolean) $enable;
    }

    /**
     * Execute the command line using 'exec' call.  STDOUT is collected in $this->data and the return code value is
     * captured in $this->result.  For Linux systems, we expect the result to be 0 on success.  Any other non-zero
     * return code is an error.
     * The return value from this function is useless since any non-zero value will trigger an Exception before the
     * value ever gets a chance to return.
     * @return int - return code (0 = success, >0 = error)
     * @throws Exception
     */
    public function execute()
    {
        // reset output
        $this->data = $this->result = false;

        // prepare command (render as command-line string)
        $command = 'LANG=' . Epygi\Locale::getLocale() . ' ' . (string) $this;
        $this->debugLog("exec [$command]");

        // run the command
        $start = microtime(true);
        exec($command, $this->data, $this->result);
        $end = microtime(true);

        // log result code and benchmark
        $this->debugLog("exec result ($this->result)");
        $this->benchmark($start, $end);

        // command-line error occurred (result is non-zero)
        // grep returns "1" if not found
        if ($this->result > 1) {
            throw new Exception('shell command failed: ' . $this->result);
        }

        // result code
        return $this->result;
    }

    /**
     * Adds a flag with an argument to the command line.  If the argument is empty and a default alternative exists,
     * the default will be used.  If condition is false, all values are ignored (like optif below,
     * but with support for default).
     *
     * @param $flag
     * @param null $param
     * @param null $default
     * @param bool $condition
     */
    public function opt($flag, $param = null, $default = null, $condition = true)
    {
        // skip unless the condition is true
        if (!$condition) {
            return;
        }

        // add the flag without escaping it
        array_push($this->opts, $flag);

        // add parameter with proper escaping
        $this->arg($param, $default);
    }

    /**
     * Shorthand way to wrap an option=value argument set with an 'if' statement.  The flag and parameter will only
     * be added if the condition is true.
     *
     * @param $flag
     * @param null $param
     * @param null $test
     */
    public function optif($flag, $param = null, $test = null)
    {
        $param = trim($param);

        // nothing to add
        if (!$test && !$param) {
            return;
        }

        // add the flag without escaping it
        array_push($this->opts, $flag);

        // push the parameter on as an argument
        $this->arg($param);
    }

    /**
     * Adds a unix pipe character, '|', and begins a new command.
     */
    public function pipe($command)
    {
        // add pipe bar
        if (count($this->opts)) {
            $this->opts[] = '|';
        }

        // start the next command
        $this->opts[] = $command;
    }

    /**
     * Enables setting an argument without escaping.
     */
    public function raw($text)
    {
        $this->opts[] = $text;
    }

    public function stderrToStdin()
    {
        $this->opts[] = '2>&1';
    }

    /**
     * Enable verbose debugging output.
     */
    public static function verbose($enable = true)
    {
        self::$verbose = (boolean) $enable;
    }
}