<?php
namespace Epygi\Storage;

use Exception;

class ConfigParser
{
    // absolute file path
    private $abs_file_path = null;

    // current line number we are processing
    private $line_number = -1;

    /**
     * @param $abs_file_path
     */
    public function __construct($abs_file_path)
    {
        $this->abs_file_path = $abs_file_path;
    }

    /**
     * Parse the config file into an array of ConfigSection objects.  We will use these sections in the ConfigFile to
     * access the config contents.
     * @return array - array of ConfigSection objects
     * @throws Exception
     */
    public function parse()
    {
        // make sure file exists
        if (!file_exists($this->abs_file_path)) {
            throw new Exception(sprintf('file does not exist "%s"', $this->abs_file_path));
        }

        // open file for reading
        $fh = fopen($this->abs_file_path, 'r');
        if (!$fh) {
            throw new Exception(sprintf('can not read file "%s"', $this->abs_file_path));
        }

        // create the "ROOT" section and make it the first (and mandatory) section in our section list
        $current = $root = new ConfigSection('ROOT');
        $sections = array($root);

        // read one line at a time into buffer
        $this->line_number = 0;
        while (!feof($fh)) {
            $this->line_number++;

            // read line from file
            $line = trim(fgets($fh));

            // blank line
            if (!$line) {
                continue;
            }

            // comment
            elseif (preg_match('/^\s*#|^\s*\/\//', $line)) {
                continue;
            }

            // start section
            elseif (preg_match('/^(\S+)?\s*(.*)\s*{/', $line, $regs)) {
                $label = trim($regs[1]);
                $name = trim($regs[2]);

                // oops, we got a label but not a name
                if ($label && !$name) {
                    $name = $label;
                    $label = '';
                }

                // we do not support nested sections
                if ($current !== $root) {
                    $this->warn('section "%s" already started.  found new section "%s"', $current->getName(), $name);
                }

                // create a new block and add it to our list
                $current = new ConfigSection($name, $label);

                // save all sections in order they appear in config file
                $sections[] = $current;
                continue;
            }

            // end section
            elseif (preg_match('/^\s*}\s*$/', $line, $regs)) {
                // we are not inside a section (we are in the root)
                if ($current == $root) {
                    $this->warn('found section end without section start');
                }

                // end section
                $current = $root;
                continue;
            }

            // key=value pair
            elseif (preg_match('/^\s*(.+?)\s*=(.*)$/', $line, $regs)) {
                $key = trim($regs[1]);
                $value = trim($regs[2]);

                // save key/value pair
                $current->$key = $value;
                continue;
            }

            // WTF?
            else {

                $this->warn('unknown line format');
                continue;
            }
        }

        fclose($fh);

        // return file config data
        return $sections;
    }

    private function warn($format)
    {
        $args = func_get_args();
        $message = call_user_func_array('sprintf', $args);

        // append current line number
        $message = sprintf('%s at line %d', $message, $this->line_number);

        // display warning message
        error_log($message);
    }
}