<?php
namespace Mvc;

use Mvc\Exception;

class Controller
{
    protected $template = 'main';

    // dependency container
    protected $front;
    protected $view;

    public function __construct($front)
    {
        $this->front = $front;
    }

    /**
     * This method is called before the action.
     */
    public function _init()
    {
        // noop
    }

    /**
     * Formats a string from a URI into a PHP-friendly name.  By default, replaces words separated by the word
     * separator character(s) with camelCaps. If $isAction is false, it also preserves replaces words separated by
     * the path separation character with an underscore, making the following word Title cased. All non-alphanumeric
     * characters are removed.
     *
     * @param string $name
     *
     * @return string
     */
    private function formatNameCamelCase($name)
    {
        // break apart word boundaries
        $words = str_replace(array('-', '.'), ' ', strtolower($name));

        // remove all other, non-word characters
        $words = preg_replace('/[^a-z0-9 ]/', '', $words);

        // uppercase words and join them
        return lcfirst(str_replace(' ', '', ucwords($words)));
    }

    /**
     * Read a param from the Front/Request object
     *
     * @param $key
     * @param $default null
     *
     * @return mixed
     */
    public function param($key, $default = null)
    {
        return $this->front->param($key, $default);
    }

    /**
     * Execute the 'action' method by name and return the rendered template view.
     * @return string
     * @throws Exception\PageNotFoundException
     */
    public function run()
    {
        // controller and action
        $controller_name = $this->param('controller');
        $action_name = $this->param('action');

        // controller and action names for view scripts
        $v_controller = strtolower($controller_name);
        $v_action = strtolower($action_name);

        // action method
        $action_method = $this->formatNameCamelCase($action_name, true) . 'Action';

        // action method not found in controller class
        if (!method_exists($this, $action_method)) {
            throw new Exception\PageNotFoundException("Method {$action_method} does not exist in " . get_class($this));
        }

        // TEMPLATE VIEW - assign template 'layout'
        $this->view = new View();

        // always do this before our action
        $this->_init();

        // execute action
        $params = $this->$action_method();

        // merge params into our view script variables
        if ($params) {
            $this->view->mergeParams($params);
        }

        // CONTENT VIEW - assign content view script
        $content = new View();
        $content->setView($v_controller . '/' . $v_action);

        // layout view
        $this->view->setLayout($this->template);

        // render content, assign content to template, then render the template
        $this->view->content = $content->render();
        return $this->view->render();
    }

    /**
     * Chooses a template layout to use.
     *
     * @param $template - name of template to use (default to main if no name provided)
     */
    function setTemplate($template = 'main')
    {
        $this->template = $template ? : 'main';
    }
}