<?php
namespace Epygi\Db;

use Epygi\Object;
use Exception;
use Mvc;
use PDO;

abstract class PdoConnection
{
    const DEBUG = false;

    /**
     * Start a new transaction.
     * @return bool
     */
    public static function beginTransaction()
    {
        try {
            // fetch connection
            $pdo = static::getConnection();

            // begin transaction
            $pdo->beginTransaction();
            return true;
        }
        catch (Exception $ex) {
            // that didn't work
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * Commit current transaction.
     * @return bool
     */
    public static function commit()
    {
        try {
            // fetch connection
            $pdo = static::getConnection();

            // commit
            $pdo->commit();
            return true;
        }
        catch (Exception $ex) {
            // that didn't work
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * Database failed, debug backtrace into error logs.
     *
     * @param $message
     *
     * @return bool
     */
    public static function dbFail($message)
    {
        edump(str_repeat('-', 50), 'DB ERROR: ' . $message);

        $backtrace = debug_backtrace();
        $traces = array();
        foreach ($backtrace as $index => $bt) {
            if (is_null($bt)) {
                continue;
            }

            // data
            $traces[] = $trace = sprintf(
                'BT-%02d %s:%d called %s%s%s',
                $index + 1,
                empty($bt['file']) ? '<no file>' : $bt['file'],
                empty($bt['line']) ? '<no line>' : $bt['line'],
                empty($bt['class']) ? '' : $bt['class'],
                empty($bt['type']) ? '' : $bt['type'],
                empty($bt['function']) ? '<no function>' : $bt['function']
            );
            edump($trace);
        }

        return false;
    }

    /**
     * Find our PDO connection and execute the SQL statement.  Return the statement handler so we can iterate
     * through the results efficiently.  Most of the functions in this class used this method, so this is a good
     * place to do some benchmarking.
     *
     * @param Statement $SQL
     *
     * @return \PDOStatement
     * @throws Exception
     */
    public static function execute(Statement $SQL)
    {
        // get connection
        $pdo = static::getConnection();

        // execute the statement
        $start = microtime(true);
        $sth = $SQL->execute($pdo);

        // debugging
        if (self::DEBUG) {
            $end = microtime(true);
            $message = sprintf('%0.4f %s', $end - $start, $SQL->getDebugLine());
            edump($message);
        }

        // return statement handle
        return $sth;
    }

    /**
     * @return PDO
     * @throws Exception
     */
    public static function getConnection()
    {
        throw new Exception ('abstract method!  See child class!');
    }

    /**
     * Prepare a statement and return the handle so raw PDO commands can be executed against it.
     *
     * @param Statement $SQL
     *
     * @return \PDOStatement
     * @throws Exception
     */
    protected static function prepare(Statement $SQL)
    {
        // get connection
        $pdo = static::getConnection();

        return $SQL->prepare($pdo);
    }

    /**
     * @param Statement $SQL
     *
     * @return bool|int - number of affected rows
     */
    private static function queryAffectedRows(Statement $SQL)
    {
        try {
            // execute the statement
            $sth = self::execute($SQL);

            // return number of rows affected
            return $sth->rowCount();
        }
        catch (Exception $ex) {
            // that didn't work (catch PDOException also)
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * @param Statement $SQL
     *
     * @return bool|int - number of affected rows
     */
    public static function queryDelete(Statement $SQL)
    {
        return self::queryAffectedRows($SQL);
    }

    /**
     * @param Statement $SQL
     *
     * @return bool|string - last inserted id
     */
    public static function queryInsert(Statement $SQL)
    {
        try {
            // execute the statement
            $sth = self::execute($SQL);

            // insert failed
            if (!$sth->rowCount()) {
                return false;
            }

            // look up the last inserted id
            $pdo = static::getConnection();
            $insert_id = $pdo->lastInsertId();

            // SQLite and MySQL
            return $insert_id;
        }
        catch (Exception $ex) {
            // that didn't work (catch PDOException also)
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * @param Statement $SQL
     *
     * @return bool|int - number of affected rows
     */
    public static function queryUpdate(Statement $SQL)
    {
        return self::queryAffectedRows($SQL);
    }

    /**
     * Roll back current transaction.
     * @return bool
     */
    public static function rollBack()
    {
        try {
            $pdo = static::getConnection();

            // roll back
            $pdo->rollBack();
            return true;
        }
        catch (Exception $ex) {
            // that didn't work
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * Executes SQL and returns array of associative arrays.
     *
     * @param Statement $SQL
     * @param string $key_key
     * @param string $value_key
     *
     * @return array|bool
     */
    public static function selectMany(Statement $SQL, $key_key = '', $value_key = '')
    {
        $data = array();
        try {
            // execute the statement
            $sth = self::execute($SQL);

            // repackage the results
            while ($row = $sth->fetch(PDO::FETCH_ASSOC)) {
                // no key_key or value_key... save whole row enumerated
                if (!$key_key) {
                    $data[] = $row;
                }
                // key_key but no value_key... save whole row with key_key
                elseif (!$value_key) {
                    $data[$row[$key_key]] = $row;
                }
                // key_key and value_key... save value_key with key_key
                else {
                    $data[$row[$key_key]] = $row[$value_key];
                }
            }
            return $data;
        }
        catch (Exception $ex) {
            // that didn't work (catch PDOException also)
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * Select data as a multi-level nested array.  Parameters are the columns to
     * use as keys in the returned array.  Multiple keys accepted.
     *
     * @param Statement $SQL
     *
     * @return array|bool
     */
    public static function selectNested(Statement $SQL)
    {
        // start with empty return data
        $data = array();

        // indexes
        $args = func_get_args();
        array_shift($args);

        try {
            // execute the statement
            $sth = self::execute($SQL);

            // loop through all results as associative array
            while ($row = $sth->fetch(PDO :: FETCH_ASSOC)) {
                $node = &$data;
                foreach ($args as $arg) {
                    $key = $row[$arg];
                    if (!isset ($node[$key])) {
                        $node[$key] = array();
                    }
                    // move to the new node
                    $node = &$node[$key];
                }

                // save the row
                $node = $row;
            }

            // success!
            return $data;
        }
        catch (Exception $ex) {
            // that didn't work (catch PDOException also)
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * Executes SQL and returns first row of results as an associative array.
     *
     * @param Statement $SQL
     *
     * @return bool|mixed
     */
    public static function selectRow(Statement $SQL)
    {
        try {
            // execute the statement
            $sth = self::execute($SQL);

            // return the first row fetched
            return $sth->fetch(PDO::FETCH_ASSOC);
        }
        catch (Exception $ex) {
            // that didn't work (catch PDOException also)
            return self::dbFail($ex->getMessage());
        }
    }

    /**
     * @param Statement $SQL
     *
     * @return Object - object wrapper around selected row
     */
    public static function selectRowObj(Statement $SQL)
    {
        $data = self::selectRow($SQL);
        return new Object($data);
    }

    /**
     * Returns the value from a specific column on the first row of the query results.
     *
     * @param Statement $SQL
     * @param $column
     * @param bool $default
     *
     * @return bool
     */
    public static function selectValue(Statement $SQL, $column, $default = false)
    {
        $row = self::selectRow($SQL);
        return isset ($row[$column]) ? $row[$column] : $default;
    }

    /**
     * @return Statement
     */
    public static function statement()
    {
        return new Statement();
    }
}