<?php
namespace Mvc;

class Param
{
    /**
     * Fetch and return an element from the Request array and return it.  If it is not set, use the default value.
     */
    public static function request($key, $default = false)
    {
        return isset ($_REQUEST[$key]) ? $_REQUEST[$key] : $default;
    }

    /**
     * Fetch an element from the Request array and return it as a boolean value.
     */
    public static function requestBool($key, $default = false)
    {
        $v = self::request($key, $default);

        // we can only test scalar values
        if (!is_scalar($v)) {
            return (boolean) $v;
        }

        // case-insensitive checking
        $v = strtolower($v);

        // clearly it is TRUE
        if ($v == 'yes' || $v == 'true' || $v == 't' || $v == 'on' || $v === true) {
            return true;
        }

        // clearly it is FALSE
        if ($v == 'no' || $v == 'false' || $v == 'f' || $v == 'off' || $v === false) {
            return false;
        }

        // its not clear, just cast it and be done with it
        return (boolean) $v;
    }

    /**
     * Safely fetch a value from the PHP superglobal $_SERVER without throwing a PHP warning about undefined keys.
     * If the key does not exist or is empty, we will return a default value of our choosing or false otherwise.
     */
    public static function server($key, $default = null)
    {
        return empty ($_SERVER[$key]) ? $default : $_SERVER[$key];
    }

    /**
     * This should never be needed but was created to perform the opposite function of a 'get'.  On some
     * environments, it may be necessary to override system defaults by setting $_SERVER values manually.
     */
    public static function serverSet($key, $value)
    {
        $_SERVER[$key] = $value;
    }

    public static function getRemoteIpAddress() {

        // Check for shared Internet/ISP IP
        if (!empty($_SERVER['HTTP_CLIENT_IP']) && self::validateIP($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        }

        // Check for IP addresses passing through proxies
        if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {

            // Check if multiple IP addresses exist in var
            if (strpos($_SERVER['HTTP_X_FORWARDED_FOR'], ',') !== false) {
                $iplist = explode(',', $_SERVER['HTTP_X_FORWARDED_FOR']);
                foreach ($iplist as $ip) {
                    if (self::validateIP($ip))
                        return $ip;
                }
            }
            else {
                if (self::validateIP($_SERVER['HTTP_X_FORWARDED_FOR']))
                    return $_SERVER['HTTP_X_FORWARDED_FOR'];
            }
        }
        if (!empty($_SERVER['HTTP_X_FORWARDED']) && self::validateIP($_SERVER['HTTP_X_FORWARDED']))
            return $_SERVER['HTTP_X_FORWARDED'];
        if (!empty($_SERVER['HTTP_X_CLUSTER_CLIENT_IP']) && self::validateIP($_SERVER['HTTP_X_CLUSTER_CLIENT_IP']))
            return $_SERVER['HTTP_X_CLUSTER_CLIENT_IP'];
        if (!empty($_SERVER['HTTP_FORWARDED_FOR']) && self::validateIP($_SERVER['HTTP_FORWARDED_FOR']))
            return $_SERVER['HTTP_FORWARDED_FOR'];
        if (!empty($_SERVER['HTTP_FORWARDED']) && self::validateIP($_SERVER['HTTP_FORWARDED']))
            return $_SERVER['HTTP_FORWARDED'];

        // Return unreliable IP address since all else failed
        return $_SERVER['REMOTE_ADDR'];
    }

/**
 * Ensures an IP address is both a valid IP address and does not fall within
 * a private network range.
 */
public static function validateIP($ip) {
           return filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6);
    }
}