<?php
namespace Mvc;

use Epygi;
use Epygi\Enum\Role as ERole;
use Epygi\Selected;
use Epygi\Help;
use Mvc;
use Mvc\Exception\PageNotFoundException;

class Router
{
    /**
     * This file will be responsible for reading the cgilist configuration and will know everything about page
     * routing including: roles, permissions, tree structure, page titles, etc.
     */

    // relative to ROOT_DIR
    const CGILIST_CONFIG_FILE = '/assets/config/cgi-list.php';
    const LEGACY_CONFIG_FILE = '/assets/config/legacy.php';

    private $cgi_list = array();

    private $selected_uri = null; // page requested by browser
    private $wrapped_uri = null; // page sub-requested by wrapper
    private $wrapped_params = array();

    private $crumbs = array();
    private $current = array();

    // map an array of children for each parent (helps to build trees)
    private $children = array();

    // map and array of cgi paths to cgi_names
    private $path_to_cginame = array();

    public function __construct()
    {
        // load data
        $data = include ROOT_DIR . self::CGILIST_CONFIG_FILE;

        /*
         * if the "condition" field exists on this route rule, and that condition is FALSE, then this rule
         * should be removed from the system.  This allows all rules to be defined in one "cgilist" file,
         * but some rules are deactivated on specific hardware based on Features.
         *
         * IMPORTANT NOTE: By removing the routes here, we do so before the tree structure is formed!  That means
         * that you will be lopping off entire tree branches if a parent branch has a false condition.  The inner
         * leaf nodes will still "route" from the URL if their conditions are not false, but they will not appear
         * in tabs, pills, and page highlighting/hierarchy.
         */
        $this->cgi_list = array_filter(
            $data,
            function ($route) {
                return (isset($route['condition']) && !$route['condition']) ? false : true;
            }
        );

        // iterate through all cgilist routes and generate parent/child relationships
        foreach ($this->cgi_list as $cgi_name => $data) {
            /**
             * To create menu trees, routes have been organized into parent/child relationships.  Each route
             * should have a parent (but it is not required).  Root menu items will have a parent = "ROOT".
             * If the route does not have a parent, we will be unable to highlight the proper menu items to
             * know where the page fits into the menu hierarchy.
             */
            if (isset($data['parent'])) {
                $parent = $data['parent'];

                // make sure the parent array index has been created
                if (!isset($this->children[$parent])) {
                    $this->children[$parent] = array();
                }

                // add this child to our parent array
                $this->children[$parent][] = $cgi_name;
            }

            /**
             * More than one cgi_name "page" can have the same path.  Here we map the multiple cgi_names onto the
             * path to make matching faster.
             */
            $path = parse_url($cgi_name, PHP_URL_PATH);
            $this->path_to_cginame[$path][] = $cgi_name;
        }
    }

    /**
     * @param $in_routes - array of routes to filter
     * @param $acl - target acl we are seeking
     *
     * @return array - filtered array
     */
    private function aclFilterRoutes($in_routes, $acl)
    {
        $routes = array();
        foreach ($in_routes as $route_name => $route) {
            // if the selected route does matches our target acl, keep the route
            if ($this->aclHasPermission($route_name, $acl)) {
                $routes[$route_name] = $route;
            }
        }
        return $routes;
    }

    /**
     * Recursively look up the cgi list tree and find an 'acl' parameter.  Return whether the logged in user has
     * permission to the route.  If no 'acl' parameter is found, access is DENIED (most restrictive).
     *
     * @param $route_name - name of route
     * @param null $target_acl - acl permission to test (default to auth role if null)
     *
     * @return bool - whether the route recursively has the ACL permission
     */
    private function aclHasPermission($route_name, $target_acl = null)
    {
        // read the acl for this route recursively
        $route_acl = $this->getAttributeRecursive('acl', $route_name);

        // if no acl is defined, DENY
        if ($route_acl === false) {
            return false;
        }

        // we have an ACL, so we need to make sure the logged in user is in the ACL rights
        $auth_acl = $target_acl ? : Epygi\Auth::getRole();
        return (boolean) ($route_acl & $auth_acl);
    }

    /**
     * We have an array of 'crumbs' which are the ancestors of the current page (including the page itself) in
     * the revers order as they appear in the parent tree.  The route 3 back from the end is our level 3 link.
     * If our level 3 link is at the begining of our list, we are not deep enough to need a link.
     * @return array|null - route to level 3
     */
    public function findLevel3IfDeeper()
    {
        // how many links are in our crumb trail?  Find position of level 3 (from end of array)
        $level3_crumb_pos = count($this->crumbs) - 3;

        // return level 3 link found or null if not deep enough yet
        if ($level3_crumb_pos >= 1) {
            $route_name = $this->crumbs[$level3_crumb_pos];
            $route = $this->cgi_list[$route_name];
            $route['route_name'] = $route_name;
            return $route;
        }

        // not deep enough to need level 3 route
        return null;
    }

    /**
     * Look up the ACL rights from the /tmp/acldb file which match this CGI name.  Always allow admin to access
     * everything, but possibly add additional roles access if the keywords are set for them.
     *
     * @param $cgi_name - name of the CGI
     *
     * @return int
     */
    public function getAcl($cgi_name)
    {
        // open acldb and read/cache the "rights" data
        static $rights = null;
        if (!$rights) {
            $cfg = Epygi\Storage\ConfigFactory::acldb();
            $rights = $cfg['rights'];
        }

        // read rights for given cgi name
        $cgi_rights = array_unique(preg_split('/\s*,\s*/', $rights->get($cgi_name), 0, PREG_SPLIT_NO_EMPTY));

        // admin can do everything!
        $flag = Epygi\Enum\Role::ADMIN;

        // translate right name into ROLE enum value
        foreach ($cgi_rights as $right) {
            switch ($right) {
                case 'localadmins':
                    $flag += Epygi\Enum\Role::LOCALADMIN;
                    break;

                case 'extensions':
                    $flag += Epygi\Enum\Role::EXTENSION;
                    break;

                case 'moderators':
                    $flag += Epygi\Enum\Role::CONFERENCE;
                    break;

                case 'superadmins':
                    $flag += Epygi\Enum\Role::SUPERADMIN;
                    break;

                case 'voicemailbox':
                    $flag += Epygi\Enum\Role::VOICEMAILBOX;
                    break;
            }
        }

        // return role enum for this cgi ACL
        return $flag;
    }

    public function getAttributeRecursive($attr_name, $cgi_name = null)
    {
        // use current cginame if no cgi name provided
        $cgi_name = $cgi_name ? : $this->selected_uri;

        // invalid cgi name, stop searching
        if (!isset($this->cgi_list[$cgi_name])) {
            return false;
        }

        // if the attribute is set on this route, use it ... done
        if (isset($this->cgi_list[$cgi_name][$attr_name])) {
            return $this->cgi_list[$cgi_name][$attr_name];
        }

        // if we have a parent route, recursively check if the value was set there
        $parent = isset($this->cgi_list[$cgi_name]['parent']) ? $this->cgi_list[$cgi_name]['parent'] : false;
        if ($parent && $parent !== 'ROOT') {
            return $this->getAttributeRecursive($attr_name, $parent);
        }

        // attribute not found!
        return false;
    }

    public function getCgiByHelpRoot($helpRoot)
    {
        if(!empty($helpRoot)) {
            foreach ($this->cgi_list as $cginame => $data) {
                if(isset($data['condition']) && !$data['condition']) {
                    continue;
                }
                if(!Help::hasHelp($cginame)) {
                    continue;
                }
                $help_url = Help::getHelpUrl($cginame);
                if(!empty($help_url) && $helpRoot === $help_url)
                {
                    return "/" . $cginame;
                }
            }
        }
        return "";
    }

    private function getChildDeepSrc($child)
    {
        $route = $this->cgi_list[$child];
        $type = isset($route['type']) ? $route['type'] : '';

        // not a container, use it as-is
        if (!in_array($type, array('tabs', 'pills', 'box'))) {
            return $child;
        }

        // recursively find the container's src
        $children = isset($this->children[$child]) ? $this->children[$child] : array();
        $first = array_shift($children);
        return $first ? $this->getChildDeepSrc($first) : '';
    }

    /**
     * TODO: make sure this list of children has been restricted to only the menu items which match our ACL, ROLE,
     * and MENU settings.
     *
     * @param $parent - parent route ("ROOT" for top level)
     *
     * @return array - array of child routes
     */
    public function getChildren($parent)
    {
        // we have all our children indexed already ... how convenient
        $children = isset($this->children[$parent]) ? $this->children[$parent] : array();

        // collect route data for each of these children
        $data = array();
        foreach ($children as $child) {
            $route = $this->cgi_list[$child];

            // only load routes which are visible in the menu
            if (!$this->getAttributeRecursive('menu', $child)) {
                continue;
            }

            // remove route which user does not have ACL permission for
            if (!$this->aclHasPermission($child)) {
                continue;
            }

            // dig past menus to find first child src link
            $src = $this->getChildDeepSrc($child);

            // save this child record
            if(isset($route['collapsible'])) {
                $data[$child] = array(
                    'desc' => isset($route['desc']) ? $route['desc'] : '',
                    'icon' => isset($route['icon']) ? $route['icon'] : '',
                    'label' => isset($route['label']) ? $route['label'] : '',
                    'on' => isset($route['on']) ? $route['on'] : false,
                    'src' => '/' . $src,
                    'type' => isset($route['type']) ? $route['type'] : '',
                    'collapsible' => isset($route['collapsible']) ? $route['collapsible'] : ''
                );
            }else{
                $data[$child] = array(
                    'desc' => isset($route['desc']) ? $route['desc'] : '',
                    'icon' => isset($route['icon']) ? $route['icon'] : '',
                    'label' => isset($route['label']) ? $route['label'] : '',
                    'on' => isset($route['on']) ? $route['on'] : false,
                    'src' => '/' . $src,
                    'type' => isset($route['type']) ? $route['type'] : '',
                );
            }
        }
        return $data;
    }

    /**
     * We only want to load and parse the cgilist once, so find the singleton instance of this router and cache it
     * for faster lookups.
     *
     * @param bool $force_reload - create new instance (needed if auth level changes)
     *
     * @return Router
     */
    public static function getInstance($force_reload = false)
    {
        static $router = null;
        return ($router && !$force_reload) ? $router : $router = new self();
    }

    public function getLegacyClassName()
    {
        $uri = $this->selected_uri;
        return 'legacy-' . preg_replace('#\W+#', '-', $uri);
    }

    /**
     * Find Level 1 "ROOT" routes.  These are the top-level routes who's parent is ROOT.
     */
    public function getLevel1Routes($force_menu_lock = false)
    {
        $routes = $this->getChildren('ROOT');

        // if the menu lock is turned ON, remove menu items which do not match the selected ACL
        if ($force_menu_lock || $this->getAttributeRecursive('menu-lock')) {
            // filter the level 1 routes based on selected user extension
            $selected = Selected\Extension::getInstance();

            // selected a conference and we are locking the menu
            if ($selected->isConference()) {
                $routes = $this->aclFilterRoutes($routes, ERole::CONFERENCE);
            } // selected an extension and we are locking the menu
            elseif ($selected->isExtension() || $selected->isTemplate()) {
                $routes = $this->aclFilterRoutes($routes, ERole::EXTENSION);
            } elseif ($selected->isRecording()) {
                $routes = $this->aclFilterRoutes($routes, ERole::RECORDING);
            } elseif ($selected->isVoiceMailBox()) {
                $routes = $this->aclFilterRoutes($routes, ERole::VOICEMAILBOX);
            }
        } // remove menu-lock tagged root routes if menu lock is not on
        else {
            foreach ($routes as $route_name => $route) {
                if ($this->getAttributeRecursive('menu-lock', $route_name)) {
                    unset($routes[$route_name]);
                }
            }
        }

        return $this->memoryApplyToRoutes($routes);
    }

    /**
     * Find Level 2 "tabs" container routes.
     *
     * @param bool $use_memory_cached_routes - should we use memorized states to replace routes?
     *
     * @return array - routes
     */
    public function getLevel2Routes($use_memory_cached_routes = true)
    {
        $routes = $this->getRouteByContainerType('tabs');

        // for the Overview page, we want the original routes, not the state-memorized ones
        if (!$use_memory_cached_routes) {
            return $routes;
        }

        // use state-memorization to rewrite routes
        return $this->memoryApplyToRoutes($routes);
    }

    /**
     * Find Level 3 "pills" container routes.
     * @return array - routes
     */
    public function getLevel3Routes()
    {
        $routes = $this->getRouteByContainerType('pills');

        /**
         * SPR PG-18396: If we only have 1 route at level 3 (pills), likely this route is going to be the same link as the one we have
         * at level 2 (tabs), so we should just drop the links and not draw the level 3 nav bar.
         */
        return count($routes) == 1 ? array() : $routes;
    }

    /**
     * Read the page title from our 'label' field in the cgi-list.
     */
    public function getPageTitle()
    {
        return isset($this->current['label']) ? $this->current['label'] : _('Unknown Page Title');
    }

    /**
     * cleanly fetch a property from the selected route properties
     */
    public function getProperty($key)
    {
        return isset($this->current[$key]) ? $this->current[$key] : null;
    }

    /**
     * Scan through the parent breadcrumbs and find a menu node which has the matching type.  The matched node will
     * be the "container" node for our menu links.  For that node, return all of the children menu links.
     *
     * @param $seek_type - type "tabs", "pillbox", etc
     *
     * @return array - array of menu items, false if no matched nodes or no children of matched node
     */
    private function getRouteByContainerType($seek_type)
    {
        $found = false;

        // scan our crumbs and look for the matching type
        foreach ($this->crumbs as $node) {
            $type = isset($this->cgi_list[$node]['type']) ? $this->cgi_list[$node]['type'] : false;
            if ($type == $seek_type) {
                $found = $node;
                break;
            }
        }

        // did not find any nodes
        if (!$found) {
            return false;
        }

        // find all the children of this found node
        return $this->getChildren($found);
    }

    public function getRouteRecursive($cgi_name = null)
    {
        // use current cginame if no cgi name provided
        $cgi_name = $cgi_name ? : $this->selected_uri;

        // invalid cgi name, stop searching
        if (!isset($this->cgi_list[$cgi_name])) {
            return array();
        }

        // our route settings
        $route = $this->cgi_list[$cgi_name];

        // get our parent route
        $parent = isset($this->cgi_list[$cgi_name]['parent']) ? $this->cgi_list[$cgi_name]['parent'] : false;
        $parent_route = ($parent && $parent !== 'ROOT') ? $this->getRouteRecursive($parent) : array();

        // merge our route data with our parent route's data
        return array_merge($parent_route, $route);
    }

    /**
     * @return string - selected URI (no leading "/" prefix)
     */
    public function getSelectedUri()
    {
        return $this->selected_uri;
    }

    /**
     * @return string - uri of current page (without a CGI extension)
     */
    public function getUri()
    {
        return '/' . $this->selected_uri;
    }

    public function getUriWrapped()
    {
        return '/' . ($this->wrapped_uri ? : $this->selected_uri);
    }

    public function getWrappedParams()
    {
        return $this->wrapped_params;
    }

    private function memoryApplyToRoutes($routes)
    {
        // we can't do anything unless it's an array of values!
        if (!is_array($routes)) {
            return $routes;
        }

        // use the same session object for the duration of the page request
        static $session = null;
        if (!$session) {
            $session = new Mvc\Session(__CLASS__);
        }

        // find current URL, but don't go lower than a level 3 nav link
        $urls = array_slice($this->crumbs, -3);
        $memorizable_url = '/' . $this->getChildDeepSrc(array_shift($urls));

        // iterate through our routes and replace the 'src' with the memorized value
        foreach ($routes as $route => $rec) {
            // if the link is "on" we need to memorize this link by saving our current page in the session
            if ($rec['on']) {
                $session->set($route, $memorizable_url);
            }

            // load the position of our menu
            $src = $session->get($route, false);
            if ($src) {
                $routes[$route]['src'] = $src;
            }
        }

        // return updated routes (containing replaced "src" values)
        return $routes;
    }

    private function routeCgiPathBestMatch($cgi_path)
    {
        $default = null;

        // find the cgi_name with best match (considering REQUEST params)
        foreach ($this->path_to_cginame[$cgi_path] as $cgi_name) {
            // if the cgi list is marked as a default, save it in case no other matches are found
            if (isset($this->cgi_list[$cgi_name]['default']) && $this->cgi_list[$cgi_name]['default']) {
                $default = $cgi_name;
                continue;
            }

            // turn cgi_name query params into an array
            $query = parse_url($cgi_name, PHP_URL_QUERY);
            parse_str($query, $params);

            // if we don't have a default yet and there are no parameters, this can be a default
            if (!$default && !$params) {
                $default = $cgi_name;
                continue;
            }

            // don't match this cgi_name further because it would pass (no params, fall through to default later)
            if (!$params) {
                continue;
            }

            // loop through query params array and see if all the request parameter values match
            $match = true;
            foreach ($params as $key => $value) {
                if (Param::request($key) != $value) {
                    $match = false;
                    break;
                }
            }

            // all params matched, this is the best route!
            if ($match) {
                return $cgi_name;
            }
        }

        // if we got here, just use the default
        if ($default) {
            return $default;
        }

        // no default found?  That's really weird, so lets just match the first cgi_name in our array
        return $this->path_to_cginame[$cgi_path][0];
    }

    /**
     * Look in the legacy route file to find a match for this CGI name.  We need legacy routes only until the
     * product has been completely converted over to the new GUI and customers stop using the old CGI names.  So,
     * this function should be phased out over time.
     *
     * @param $cginame - route to search for in the legacy route file
     *
     * @return bool - false if not legacy route matches, redirect to new URL if route does match
     * @throws PageNotFoundException - route matches an invalidated url (we specifically stop searching on match)
     */
    private function routeLegacy($cginame)
    {
        // look up cginame in legacy rewrite file
        $legacy = include ROOT_DIR . self::LEGACY_CONFIG_FILE;
        $redir = isset($legacy[$cginame]) ? $legacy[$cginame] : null;

        // not found in legacy rewrites
        if (is_null($redir)) {
            return false;
        }

        // page has been invalidated, stop here!
        if (!$redir) {
            throw new PageNotFoundException('Page Removed!: ' . $cginame);
        }

        // redirect to the new URL
        header('Location:' . '/' . $redir);
        exit;
    }

    /**
     * Starting from our selected page, mark all pages in path (walk up the inheritance parent/child tree) as ON
     * and save the breadcrumb trail.
     */
    private function routeMarkBreadcrumbs()
    {
        // reset breadcrumb path
        $this->crumbs = array();

        // starting with our selected URI
        $current = $this->selected_uri;

        // walk up the "parent" nodes and mark all cgilist entries as ON (selected)
        while ($current && isset($this->cgi_list[$current])) {
            // save this breadcrumb
            $this->crumbs[] = $current;

            // mark this page as ON
            $this->cgi_list[$current]['on'] = true;

            // find the next page by finding our parent
            $current = isset($this->cgi_list[$current]['parent']) ? $this->cgi_list[$current]['parent'] : false;
        }
    }

    /**
     * Searches the legacy routes and cgilist routes to match both <cginame> and <controller>/<action> path formats.
     *
     * @param $uri - current page URI
     *
     * @return array - controller, action, and parameters extracted from url and from route match
     * @throws PageNotFoundException - if no routes match
     */
    public function routeMatch($uri)
    {
        // clean the uri and strip the '.cgi' extension from old CGI urls
        $cgi_path = preg_replace('#\.cgi$#', '', ltrim($uri, '/'));

        // route legacy redirects
        $this->routeLegacy($cgi_path);

        // look up the cgi_path in our path mappings
        if (!isset($this->path_to_cginame[$cgi_path])) {
            throw new PageNotFoundException('Page not found: ' . $uri);
        }

        // find the best route match for this cgi path
        $cgi_name = $this->routeCgiPathBestMatch($cgi_path);

        // after matching cgi_path to cgi_name, we'll never use this array again (let's free some memory)
        unset($this->path_to_cginame);

        // this is the currently selected ROUTE
        $this->selected_uri = $cgi_name;
        $this->current = $this->cgi_list[$cgi_name];

        // mark the current route as "on" and also mark breadcrumb "parent routes" as "on".
        $this->routeMarkBreadcrumbs();

        // study matched route and define the controller / action / and params
        return $this->routeTranslate();
    }

    /**
     * Study the cgi_name data and select the appropriate Controller, Action, and Request Parameters.
     * @return array - controller, action, params
     * @throws PageNotFoundException - no action or controller set
     */
    private function routeTranslate()
    {
	$template = null;
        // 1) if "action" is defined in the selected route rule, use this setting and search no further
        if (isset($this->current['action'])) {
            $xargs = $this->current['action'];
            $controller = array_shift($xargs);
            $action = array_shift($xargs);
            $args = (array) array_shift($xargs);
        }

        // 2) path uses controller/action format, let's use that
        elseif (preg_match('#/#', $this->selected_uri)) {
            $parts = preg_split('#/#', $this->selected_uri, 0, PREG_SPLIT_NO_EMPTY);
            $controller = array_shift($parts);
            $action = array_shift($parts);
            $args = array();
        }

        // 3) everything below is a CGI wrapped page!
        else {
            $controller = 'index';
            $action = 'cgi-wrapper';
			
	    if (isset($this->current['template'])) {
		$template = $this->current['template'];
	    }

            // turn cgi_name query params into an array
            $query = parse_url($this->selected_uri, PHP_URL_QUERY);
            parse_str($query, $args);
            $args = (array) $args;

            // these will be passed to the CGI wrapper
            $this->wrapped_uri = parse_url($this->selected_uri, PHP_URL_PATH);
            $this->wrapped_params = $args;
        }

        // convert remaining parts to key=value params
        $params = array();
        foreach ($args as $key => $value) {
            $params[$key] = $value;
        }

        // set action and controller defaults and merge into params array (do this last to prevent override)
        $params['controller'] = $controller ? : 'index';
        $params['action'] = $action ? : 'index';
	if($template){
		$params['template'] = $template;
	}

        // we found everything we needed!
        return $params;
    }
}