<?php
namespace Epygi\Locale;

use Epygi;
use Exception;
use Mvc;

class PoFile
{
    const BIN_EGREP = '/bin/egrep';
    const BIN_FIND = '/usr/bin/find';
    const BIN_MSGFMT = '/usr/bin/msgfmt';
    const BIN_XGETTEXT = '/usr/bin/xgettext';

    public static function export($locale)
    {
        // clean up locale
        $locale = preg_replace('/\.*$/', '', $locale);

        // output directory
        $dir = ROOT_DIR . '/locale/' . $locale . '/LC_MESSAGES';

        // create output directory
        umask(0);
        if (!is_dir($dir)) {
            if (!mkdir($dir, 0775, true)) {
                throw new Exception('unable to build messages output directory.');
            }
        }

        // target "PO" locale export file
        $file_name = $dir . '/messages.po';

        // open the messages.po file for writing
        $fp = fopen($file_name, 'w');
        if (!$fp) {
            throw new Exception('unable to open locale export file for writing (' . $file_name . ')');
        }

        // write PO file header
        $header = file_get_contents(ROOT_DIR . '/locale/locale_po_header.txt');
        fwrite($fp, $header);

        // fetch translated strings
        $messages = Query\Locale::msgGetTranslated();

        //----------
        // write the messages to the file
        foreach ($messages as $message => $translation) {
            // original
            $message = addcslashes($message, "\r\n\\\"");
            fwrite($fp, "msgid \"$message\"\n");

            // translation
            $translation = addcslashes($translation, "\r\n\\\"");
            fwrite($fp, "msgstr \"$translation\"\n");

            // blank line
            fwrite($fp, "\n");
        }

        // done with the file
        fclose($fp);
        chmod($file_name, 0664);

        //----------
        // file names for compilation
        $tmpfile_name = $dir . '/messages.mo.tmp';
        $outfile_name = $dir . '/messages.mo';

        // compile "messages.po" file into "messages.mo"
        $cmd = new Epygi\Shell\Command(self::BIN_MSGFMT);
        $cmd->opt('--output-file=' . $tmpfile_name);
        $cmd->arg($file_name);

        // run command
        $cmd->execute();

        // rename temp file
        if (file_exists($tmpfile_name)) {
            if (!rename($tmpfile_name, $outfile_name)) {
                throw new Exception('unable to move compiled MO file into position.');
            }
        }

        // ok
        return true;
    }

    public static function import($revision)
    {
        // run forever
        set_time_limit(0);

        // build command
        $cmd = new Epygi\Shell\Command(self::BIN_FIND);
        $cmd->arg(ROOT_DIR);
        $cmd->opt('-type', 'f');

        $cmd->pipe(self :: BIN_EGREP);
        $cmd->arg('(php|phtml)$');

        $cmd->pipe(self :: BIN_XGETTEXT);
        $cmd->opt('--from-code="UTF-8"');
        $cmd->opt('--language', 'php');
        $cmd->opt('-f', '-');
        $cmd->opt('--no-wrap');
        $cmd->opt('--omit-header');

        <?php if ( Mvc\Feature::hasIserv() ) : ?>
            $cmd->opt('--copyright-holder="Epygi Technologies"');
            $cmd->opt('--msgid-bugs-address="admin@epygi.com"');
        <?php else: ?>
            $cmd->opt('--copyright-holder="Epygi Technologies"');
            $cmd->opt('--msgid-bugs-address="admin@epygi.com"');
        <?php endif; ?>

        $cmd->opt('-o', '-'); // stdout

        Mvc\Debug::dump((string) $cmd);

        // run command
        $cmd->execute();

        exit;
        // parse results to here
        $messages = array();
        $files = array();

        // process GETTEXT search results
        foreach ($cmd->data as $line) {
            $line = trim($line);

            // skip blank lines, and translation "msgstr" lines
            if (!$line || preg_match('/^msgstr /', $line)) {
                $files = array();
                continue;
            }

            // found message ("msgid" is a new string to translate)
            if (preg_match('/^msgid /', $line)) {
                // remove prefix and surrounding quotes
                $text = preg_replace('/^msgid "|"\s*$/', '', $line);
                $text = stripcslashes($text);

                // save the text and locations
                if ($text) {
                    $messages[$text] = $files;
                }

                // start with empty files again
                $files = array();
                continue;
            }

            // found location ("#:" denotes file and line number where found)
            if (preg_match('/^\#:\s+/', $line)) {
                // extract file name and line number
                list ($junk, $file_name, $line_number) = preg_split('/:/', $line);
                $file_name = preg_replace('#^.*/world#', '', $file_name);

                // save the file name and line number
                array_push(
                    $files,
                    array(
                        $file_name,
                        $line_number
                    )
                );
            }
        }

        foreach ($messages as $text => $files) {
            // create or update message
            $result = Query\Locale::msgCreate($text, $revision);

            // remove the old locations
            Query\Locale::locaDeleteByHash($text);

            // add the new locations
            foreach ($files as $loc) {
                list ($file_name, $line_number) = $loc;
                Query\Locale::locaCreate($text, $file_name, $line_number);
            }
        }

        // ok
        return true;
    }
}