<?php
namespace Epygi\Form;

use Exception;
use Mvc;

abstract class Engine
{
    const SESSION_PREFIX = 'FORM_';
    const STEP_EXPECTED_KEY = '__expected_action';
    const STEP_STATUS_KEY = '__step_complete';
    const TOKEN_NEXT = 'qform_next';
    const TOKEN_PREVIOUS = 'qform_previous';

    /**
     * Manage multi-step forms or wizards.  This involves 1) managing the various steps 2) storing its values and 3)
     * switching between each step.  This is based on the sample code and tutorial found here:
     * - http://www.zervaas.com.au/resources/code-library/zervwizard
     * - http://www.phpriot.com/d/articles/php/application-design/multi-step-wizards/index.html
     */
    private static $instance_counter = 0;
    public $ERRORS = array();
    protected $REQUIRED = array();
    protected $FAKE_REQ = array();

    protected $CHECK = array();
    protected $steps = array();
    protected $step_partial = array();
    protected $step = '';
    protected $is_first_load = true;
    protected $is_complete = false;

    protected $_params = array();
    protected $instance_number = 0;
    protected $session = null; // value storage

    public function __construct($params = array())
    {
        // save passed in params
        $this->_params = $params;

        // iterate instances of self so we get a unique ID
        self::$instance_counter++;
        $this->instance_number = self::$instance_counter;

        // use session for form storage
        $this->session = new Mvc\Session(self::SESSION_PREFIX . get_class($this));

        // initialize class
        $this->initialize();
    }

    /**
     * PHP5 magic function to get data from the session storage.
     */
    public function __get($key)
    {
        return $this->getValue($key);
    }

    /**
     * Sets the value into the session for storage.
     */
    public function __set($key, $value)
    {
        $this->session->set($key, $value);
    }

    protected function addStep($step_name, $view_script = null)
    {
        array_push($this->steps, $step_name);

        // save the name of the view script
        if ($view_script) {
            $this->step_partial[$step_name] = $view_script;
        }
    }

    public function check()
    {
        // find the field and method name but preserve the other args
        $args = func_get_args();
        $field = array_shift($args);
        $method = array_shift($args);

        // if the method name isn't provided, use the default name
        if (!$method) {
            $method = 'check_' . $field;
        }

        // change order to method, field, args...
        array_unshift($args, $method);

        // first check defined for this field
        if (!isset ($this->CHECK[$field])) {
            $this->CHECK[$field] = array();
        }

        // add this check
        $this->CHECK[$field][] = $args;
    }

    /**
     * Removes all data from the container. This is primarily used to reset the
     * wizard data completely
     */
    public function clear()
    {
        // keep these fields after clearing
        $args = func_get_args();

        // build our values to keep
        $keep = array();
        foreach ($args as $field) {
            $keep[$field] = $this->$field;
        }

        // clear the form
        $this->session->clear();
        $this->ERRORS = array();

        // now restore the keep values
        $this->import($keep);
    }

    /**
     * Function to run once the final step has been processed and is valid. This should be overwritten in child classes
     */
    protected abstract function execute();

    public function getMarkedError($key)
    {
        return isset ($this->ERRORS[$key]) ? $this->ERRORS[$key] : '';
    }

    /**
     * Get the name of the current step
     * @return string - The name of the current step
     * @throws Exception
     */
    public function getStepName()
    {
        // the selected step is fine, return it
        if (in_array($this->step, $this->steps)) {
            return $this->step;
        }

        // no steps defined yet
        if (!count($this->steps)) {
            throw new Exception('no steps defined');
        }

        // return the first step name
        return $this->steps[0];
    }

    /**
     * Fetches a value from the session storage.
     */
    public function getValue($key, $default = null)
    {
        //Allows for multi-section array in the html form
        //Allow fields to be named field_name[setting] etc.. so to organize different sections without name conflicts, and easier of use.
        if (strpos($key, '[') !== false) {
            //grab matching field, and process the sub-key
            preg_match_all("/([A-Za-z0-9_]+)[\[.+\]]/", $key, $matches);
            $keys = $matches[1];

            $_key = $keys[0];
            $value = $this->session->get($_key, $default);
            $_key = $keys[1];
            if (count($keys) == 2) {
                if (is_array($value)) {
                    $value = isset($value[$_key]) ? $value[$_key] : null;
                    return $value;
                }
                return $value;
            }
            else if (count($keys) == 3) {
                if (!isset($value[$_key])) {
                    return;
                }
                $value = $value[$_key];
                if (is_array($value)) {
                    $_key = $keys[2];
                    $value = isset($value[$_key]) ? $value[$_key] : null;
                    return $value;
                }
                return $value;
            }
        }

        return $this->session->get($key, $default);
    }

    public function hasError($field = false)
    {
        // looking for a specific field error
        if ($field) {
            return !empty ($this->ERRORS[$field]);
        }

        // looking for any error in general
        foreach ($this->ERRORS as $field => $error) {
            if (!empty ($error)) {
                return true;
            }
        }

        // no error exists
        return false;
    }

    public function import($data, $prefix = '')
    {
        // it's not an array...crap
        if (!is_array($data)) {
            return;
        }

        // blend the array into our form data
        foreach ($data as $field => $value) {
            $key = $prefix . $field;
            $this->setValue($key, $value);
        }
    }

    private function importPostValues()
    {
        foreach ($_REQUEST as $key => $value) {
            // checkbox marker
            if (preg_match('/^cbx/', $key)) {
                $key = preg_replace('/^cbx/', '', $key);
                $this->$key = false;
                continue;
            }

            // import this key/value pair
            $this->$key = $value;
        }
    }

    /**
     * This function is called by our constructor and should be overridden by child classes.
     */
    public function initialize()
    {
        // do nothing
    }

    /**
     * Check if the form is complete. This can only be properly determined after process() has been called.
     * @return bool - True if the form is complete and valid, false if not
     */
    public function isComplete()
    {
        return $this->is_complete;
    }

    public function isFakeRequired($field)
    {
        return isset ($this->FAKE_REQ[$field]);
    }

    public function isFirstLoad()
    {
        return $this->is_first_load;
    }

    /**
     * Check if the current step is the first step.
     * @return bool - True if the current step is the first step.
     */
    public function isFirstStep()
    {
        return (count($this->steps) > 0) && ($this->step == $this->steps[0]);
    }

    /**
     * Check if the current step is the last step.
     * @return bool - True if the current step is the last step
     */
    public function isLastStep()
    {
        $steps = array_keys($this->steps);
        return count($steps) > 0 && array_pop($steps) == $this->getStepName();
    }

    public function isRequired($field)
    {
        return isset ($this->REQUIRED[$field]);
    }

    private function markErrorMessage($key, $value)
    {
        // original field has error
        $this->ERRORS[$key] = $value;

        // done
        return false;
    }

    private function markRequired($field, $error_message = '')
    {
        if (!$error_message) {
            $error_message = _('Please enter this information.');
        }

        // original field has error
        $this->REQUIRED[$field] = $error_message;
    }

    /**
     * Function to run when the form is loaded for the first time.
     */
    protected function onFirstLoad()
    {
        // this method should be overridden by child classes
    }

    public function param($key, $default = null)
    {
        return isset($this->_params[$key]) ? $this->_params[$key] : $default;
    }

    /**
     * Processes the form for the specified step. If the processed step is complete then the wizard is set to use the
     * next step. If this is the initial call to process then the wizard is set to use the first step. Once the next
     * step is determined the prepare method is called for the step. This has the method name prepare_[step name]()
     */
    public function process()
    {
        // import post values into the current session
        $this->importPostValues();
        $this->is_first_load = false;

        // on DEV, sometimes admins want to preview the "complete" page for editing
        $force_complete = Mvc\Param::request('complete') && Mvc\Config::isDev();
        if ($force_complete) {
            $this->is_complete = true;
            $this->setStepName('complete');
            return;
        }

        // wanted step
        $want_step = $this->processCalculateNextStep(true);

        // init form
        if (!$want_step) {
            $this->is_first_load = true;
            $this->onFirstLoad();
        }

        // find a step not yet processed (the first step)
        $done = array();
        $diffs = array_diff($this->steps, $done);
        $step = array_shift($diffs);

        // process each of the steps
        while ($step) {
            // this is our current step
            $this->step = $step;

            // prepare the step
            $callback = 'prepare_' . $step;
            if (method_exists($this, $callback)) {
                $this->$callback ();
            }

            // re-calculate wanted step
            $want_step = $this->processCalculateNextStep();

            // this is the step we want to stop on next
            if ($want_step == $step) {
                return;
            }

            // Check for Errors
            $req = array_keys($this->REQUIRED);
            $chk = array_keys($this->CHECK);
            $constraint_fields = array_unique(array_merge($req, $chk));

            // loop through each constrained field and validate it
            foreach ($constraint_fields as $field) {
                $this->validate($field);
            }

            // if we have errors, we need to stop on this step
            if ($this->hasError()) {
                return $step;
            }

            // find another un-processed step
            $done[] = $step;
            $diffs = array_diff($this->steps, $done);
            $step = array_shift($diffs);
        }

        // COMPLETE!  we made it all the way through
        $worked = $this->execute();

        if ($worked && !$this->hasError()) {
            $this->is_complete = true;
            $this->setStepName('complete');
        }

        return;
    }

    /**
     * Our step name is set in the 'action=****' request parameter.  The step name is the name of the step we were on
     * last and we will be changing it to the name of the step we should be on now.  If we are on the first step the
     * step name will be returned as FALSE. If the 'previous' page token is set we find the step before the requested
     * step.
     */
    private function processCalculateNextStep($detect_first = false)
    {
        // which step were we just on?
        $selected_step = Mvc\Param::request('action', false);

        // first step
        if ($detect_first && !$selected_step) {
            return false;
        }

        // convert step name to a number
        $step_number = array_search($selected_step, $this->steps, true);

        // which button was clicked?
        $want_previous = Mvc\Param::requestBool(self::TOKEN_PREVIOUS);
        $want_next = isset($_REQUEST[self::TOKEN_NEXT]);

        // step name not in our list?  initialize the form (maybe step doesn't exist yet or FIRST LOAD)
        if ($step_number === false) {
            $step_number = 0;
        }

        // move the step number backward (clicked PREVIOUS)
        elseif ($want_previous && $step_number > 0) {
            $step_number = $step_number - 1;
        }

        // move the step number forward (clicked NEXT)
        elseif ($want_next) {
            $step_number = $step_number + 1;
        }

        // this is the step we want to display
        return isset ($this->steps[$step_number]) ? $this->steps[$step_number] : 'complete';
    }

    public function req($field, $error_message = '')
    {
        $this->markRequired($field, $error_message);
    }

    /**
     * Without using special error messages, iterate through all fields passed in and mark them as required.  This is
     * shorthand to force many fields to be required when a generic message will suffice.
     */
    public function reqAll()
    {
        // fields passed in
        $args = func_get_args();

        // loop through all fields and mark as required
        foreach ($args as $arg) {
            $this->req($arg);
        }
    }

    public function reqAllFake()
    {
        // fields passed in
        $args = func_get_args();

        // loop through all fields and mark as required
        foreach ($args as $arg) {
            $this->reqFake($arg);
        }
    }

    public function reqFake($field)
    {
        $this->FAKE_REQ[$field] = true;
    }

    private function setStepName($step_name)
    {
        if ($step_name) {
            $this->step = $step_name;
            $this->step_name = $step_name;
        }
    }

    private function setValue($key, $value)
    {
        return $this->$key = $value;
    }

    public function stepEnabled($step_name)
    {
        // no such step
        if (!in_array($step_name, $this->steps)) {
            return false;
        }

        // get index of wanted step
        $want_index = array_search($step_name, $this->steps);

        // get index of current step
        $curr_index = array_search($this->step, $this->steps);

        // is the wanted step behind us or the same as current step?
        return ($curr_index >= $want_index);
    }

    public function syserr($message)
    {
        $step_name = '';
        if (func_num_args() > 1) {
            list ($step_name, $message) = func_get_args();
        }

        $this->markErrorMessage('syserr', $message);

        // this error needs to be thrown on a specific form page
        if ($step_name) {
            $this->setStepName($step_name);
        }
        return false;
    }

    public function syserrf($message)
    {
        // separate format string from other args
        $args = func_get_args();
        $message = call_user_func_array('sprintf', $args);

        // format error message
        return $this->syserr(sprintf($message, $args));
    }

    public function trip($key, $value)
    {
        return $this->markErrorMessage($key, $value);
    }

    /**
     * Runs required and checked function checks on a given field.  Returns TRUE if an error is found.  Returns FALSE
     * if there are no errors.  Multiple calls to this function will return the cached result of previous checks
     * made with the same field argument.
     */
    private function validate($field)
    {
        // don't run checks again if this field already triggered an error
        if (isset ($this->ERRORS[$field])) {
            return (boolean) $this->ERRORS[$field];
        }

        // it's required but it's blank, complain.
        if (isset ($this->REQUIRED[$field])) {
            // trim the field
            $this->setValue($field, trim($this->getValue($field)));

            // is the field empty?
            if (strlen($this->getValue($field)) < 1) {
                $this->markErrorMessage($field, $this->REQUIRED[$field]);
                return true;
            }
        }

        // Run custom validate function on this field
        if (isset ($this->CHECK[$field])) {
            foreach ($this->CHECK[$field] as $args) {
                $method = array_shift($args);
                array_unshift($args, $field);

                $callback = array();
                array_push($callback, $this, $method);
                $err = call_user_func_array($callback, $args);

                if ($err) {
                    $this->markErrorMessage($field, $err);
                    return true;
                }
            }
        }

        // hmmm, this field looks like it's valid, remember that too (cache error check)
        $this->markErrorMessage($field, '');
        return false;
    }
}