<?php

class RestAccess
{
    const DEFAULT_CURL_TIMEOUT = 60;
    const DEFAULT_CURLOPT_CONNECTTIMEOUT = 20;

    const API_URL = 'url';
    const API_USER = 'user';
    const API_PASSWORD = 'password';

    private $curl;

    private $apiUser;
    private $apiPasswd;

    public $responseHeaders = array();
    public $rawResponseHeaders = null;

    private $curlErrorCode = null;
    private $httpStatusCode = null;
    private $curlErrorMessage = null;
    private $rawResponse = null;
    private $duration = 0;

    private $headers = array();
    private $options = array();

    private $cert;

    public function __construct($headers = array(), $authInfo = array()) {

        $this->curl = curl_init();

        $this->setApiHeaders($headers);
        $this->setApiAuthentication($authInfo);
        $this->setDefaultUserAgent();
        $this->setOpt(CURLOPT_TIMEOUT, self::DEFAULT_CURL_TIMEOUT);
        $this->setOpt(CURLOPT_CONNECTTIMEOUT, self::DEFAULT_CURLOPT_CONNECTTIMEOUT);
        $this->setOpt(CURLINFO_HEADER_OUT, true);
        $this->setOpt(CURLOPT_RETURNTRANSFER, true);
        $this->setOpt(CURLOPT_HEADERFUNCTION, array($this, 'headerCallback'));
        $this->setCert();
    }

    public function __destruct() {
        if (is_resource($this->curl)) {
            curl_close($this->curl);
        }
        unset($this->options);
        unset($this->headers);
        unset($this->responseHeaders);
    }

    public function get($url, $urlData = array()) {
        $completeUrl = $this->buildURL($url, $urlData);
        $this->setOpt(CURLOPT_URL, $completeUrl);
        $this->setOpt(CURLOPT_CUSTOMREQUEST, 'GET');
        $this->setOpt(CURLOPT_HTTPGET, true);
        return $this->exec();
    }

    public function head($url, $urlData = array()) {
        $completeUrl = $this->buildURL($url, $urlData);
        $this->setOpt(CURLOPT_URL, $completeUrl);
        $this->setOpt(CURLOPT_CUSTOMREQUEST, 'HEAD');
        $this->setOpt(CURLOPT_NOBODY, true);
        return $this->exec();
    }

    public function post($url, $postData = null) {
        $completeUrl = $this->buildURL($url);
        $this->setOpt(CURLOPT_URL, $completeUrl);
        $this->setOpt(CURLOPT_CUSTOMREQUEST, 'POST');
        $this->setOpt(CURLOPT_POST, true);
        if ($postData === null) {
            $this->unsetHeader('Content-Length');
        }
        if ($postData) {
            $data = $this->buildPostData($postData);
            $this->setOpt(CURLOPT_POSTFIELDS, $data);
        }
        return $this->exec();
    }

    public function put($url, $putData = null) {
        $completeUrl = $this->buildURL($url);
        $this->setOpt(CURLOPT_URL, $completeUrl);
        $this->setOpt(CURLOPT_CUSTOMREQUEST, 'PUT');
        if ($putData) {
            $data = $this->buildPostData($putData);
            if (empty($this->options[CURLOPT_INFILE]) && empty($this->options[CURLOPT_INFILESIZE])) {
                $this->setHeader('Content-Length', (string)strlen($data));
            }
            $this->setOpt(CURLOPT_POSTFIELDS, $data);
        }
        return $this->exec();
    }

    public function patch($url, $patchData = null) {
        $completeUrl = $this->buildURL($url);
        $this->setOpt(CURLOPT_URL, $completeUrl);
        if ($patchData === null) {
            $this->unsetHeader('Content-Length');
        }
        $this->setOpt(CURLOPT_CUSTOMREQUEST, 'PATCH');
        if ($patchData) {
            $data = $this->buildPostData($patchData);
            $this->setOpt(CURLOPT_POSTFIELDS, $data);
        }
        return $this->exec();
    }

    public function delete($url, $deleteData = null) {
        $completeUrl = $this->buildURL($url);
        $this->setOpt(CURLOPT_URL, $completeUrl);
        $this->setOpt(CURLOPT_CUSTOMREQUEST, 'DELETE');
        if ($deleteData) {
            $data = $this->buildPostData($deleteData);
            $this->setOpt(CURLOPT_POSTFIELDS, $this->buildPostData($data));
        }
        return $this->exec();
    }

    private function setApiAuthentication($loginInfo) {
    		if ($loginInfo[self::API_USER] && $loginInfo[self::API_PASSWORD]) {
				    $this->setOpt(CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
				    $this->setOpt(CURLOPT_USERPWD, $loginInfo[self::API_USER] . ":" . $loginInfo[self::API_PASSWORD]);
		    }
    }

    private function setApiHeaders ($headers) {
        foreach ($headers as $key=>$val) {
          $this->setHeader($key, $val);
        }
    }

    private function setHeader ($key, $value) {
        $this->headers[$key] = $value;
        $headers = array();
        foreach ($this->headers as $key => $value) {
            $headers[$key] = $value;
        }
        $this->setOpt(CURLOPT_HTTPHEADER, array_map(function($value, $key) {
            return $key . ': ' . $value ;
        }, $headers, array_keys($headers)));
    }

    public function unsetHeader($key) {
        $this->setHeader($key, '');
        unset($this->headers[$key]);
    }

    private function setDefaultUserAgent() {
        $user_agent = 'epygi/';
        $user_agent .= ' PHP/' . PHP_VERSION;
        $curl_version = curl_version();
        $user_agent .= ' curl/' . $curl_version['version'];
        $this->setOpt(CURLOPT_USERAGENT, $user_agent);
    }

    private function headerCallback($ch, $header) {
        $this->rawResponseHeaders .= $header;
        return strlen($header);
    }

    public function setOpt($option, $value) {
        $this->options[$option] = $value;
        return curl_setopt($this->curl, $option, $value);
    }

    private function buildURL($url, $data = array()) {
        return $url . (empty($data) ? '' : '?' . http_build_query($data));
    }

    public function buildPostData($data) {
        $jsonRegExp = '/^(?:application|text)\/(?:[a-z]+(?:[\.-][0-9a-z]+){0,}[\+\.]|x-)?json(?:-[a-z]+)?/i';
        $dataStr = '';

        if (is_array($data)) {
            if (self::isArrayMultidim($data)) {
                if (isset($this->headers['Content-Type']) && preg_match($jsonRegExp, $this->headers['Content-Type'])) {
                    $jsonStr = json_encode($data);
                    if (!($jsonStr === false)) {
                        $dataStr = $jsonStr;
                    }
                } else {
                    $dataStr = self::httpBuildMultiQuery($data);
                }
            } else {
                foreach ($data as $key => $value) {
                    if (is_array($value) && empty($value)) {
                        $data[$key] = '';
                    }
                }
                if (isset($this->headers['Content-Type']) && preg_match($jsonRegExp, $this->headers['Content-Type'])) {
                    $jsonStr = json_encode($data);
                    if (!($jsonStr === false)) {
                        $dataStr = $jsonStr;
                    }
                } else {
                    $dataStr = http_build_query($data, '', '&');
                }
            }
        } else {
            $dataStr = (string)$data;
        }
        return $dataStr;
    }

    public static function httpBuildMultiQuery($data,  $key = null) {
        $query = array();
        if (empty($data)) {
            return $key . '=';
        }
        $isArrayAssoc = self::isArrayAssoc($data);
        foreach ($data as $k => $value) {
            if (is_string($value) || is_numeric($value)) {
                $brackets = $isArrayAssoc ? '[' . $k . ']' : '[]';
                $query[] = urlencode($key === null ? $k : $key . $brackets) . '=' . rawurlencode($value);
            } elseif (is_array($value)) {
                $nested = $key === null ? $k : $key . '[' . $k . ']';
                $query[] = self::httpBuildMultiQuery($value, $nested);
            }
        }
        return implode('&', $query);
    }

    public static function isArrayAssoc( $array)  {
        $keys = array_keys($array);
        foreach ($keys as $key) {
          if (is_string($key)) {
            return true;
          }
        }
        return false;
    }

    private static function isArrayMultidim( $array)  {
        if (!is_array($array)) {
            return false;
        }
        foreach ($array as $key => $val) {
          if (is_array($val)) {
            return true;
          }
        }
        return false;
    }

    private function exec() {
        $ticksEndpointStart = (int)(microtime(true)*1000);

        $rslt = array(ApiConstants::KEY_RESPONSE => array(), ApiConstants::KEY_ERROR => false, ApiConstants::KEY_ERROR_CODE => null, ApiConstants::KEY_ERROR_MESSAGE => null);

        $convertedResponse = array();
        $this->rawResponse = curl_exec($this->curl);

        $this->curlErrorCode = curl_errno($this->curl);
        $this->curlErrorMessage = curl_error($this->curl);
        $curlError = !($this->curlErrorCode === 0);

        $this->httpStatusCode = curl_getinfo($this->curl, CURLINFO_HTTP_CODE);

        $httpError = in_array(floor($this->httpStatusCode / 100), array(4, 5));
        $error = $curlError || $httpError;
        $errorCode = $error ? ($curlError ? $this->curlErrorCode : $this->httpStatusCode) : 0;

        $this->responseHeaders = $this->convertResponseHeaders();
        if ($this->rawResponse) {
            $convertedResponse = $this->convertResponse();
        }

        $httpErrorMessage = '';
        if ($error) {
            if (isset($this->responseHeaders['Status-Line'])) {
                $httpErrorMessage = $this->responseHeaders['Status-Line'];
            }
        }
        $errorMessage = $curlError ? $this->curlErrorMessage : $httpErrorMessage;

/*
echo "\n----------------- requestHeaders ----------------------\n";
$requestHeaders = curl_getinfo($this->curl, CURLINFO_HEADER_OUT);
print_r ($requestHeaders,true);
echo "\n----------------- responseHeaders ----------------------\n";
print_r ($this->responseHeaders,true);
echo "\n---------------- rawResponse -----------------------\n";
print_r($this->rawResponse,true);
echo "\n----------------- convertedResponse ----------------------\n";
print_r($convertedResponse,true);
echo "\n---------------------------------------\n";
echo "curlErrorCode = " . $this->curlErrorCode."\n";
echo "curlErrorMessage = " . $this->curlErrorMessage."\n";
echo "curlError = " . $curlError."\n";
echo "httpStatusCode = " . $this->httpStatusCode."\n";
echo "error = " . $error."\n";
echo "errorCode = " . $errorCode."\n";
echo "errorMessage = " . $errorMessage."\n";
echo "---------------------------------------\n";
die();*/

        $this->duration = (int)(microtime(true)*1000) - $ticksEndpointStart;

        $rslt[ApiConstants::KEY_ERROR] = $error;
        $rslt[ApiConstants::KEY_ERROR_CODE] = $errorCode;
        $rslt[ApiConstants::KEY_ERROR_MESSAGE] = $errorMessage;
        $rslt[ApiConstants::KEY_RESPONSE] = $convertedResponse;
        $rslt[self::API_URL] = curl_getinfo($this->curl, CURLINFO_EFFECTIVE_URL);

        return $rslt;
    }

    private function convertResponseHeaders() {
        $responseHeaderArray = array();
        $responseHeaderArray = explode("\r\n\r\n", $this->rawResponseHeaders);
        $responseHeader  = '';
        for ($i = count($responseHeaderArray) - 1; $i >= 0; $i--) {
            if (stripos($responseHeaderArray[$i], 'HTTP/') === 0) {
                $responseHeader = $responseHeaderArray[$i];
                break;
            }
        }

        $responseHeader = preg_split('/\r\n/', $responseHeader, null, PREG_SPLIT_NO_EMPTY);
        $httpHeaders = $responseHeader;
        $rawHeadersCount = count($responseHeader);

        for ($i = 1; $i < $rawHeadersCount; $i++) {
            list($key, $value) = explode(':', $responseHeader[$i], 2);
            $key = trim($key);
            $value = trim($value);
            if (isset($httpHeaders[$key])) {
                $httpHeaders[$key] .= ',' . $value;
            } else {
                $httpHeaders[$key] = $value;
            }
        }

        $httpHeaders['Status-Line'] = isset($responseHeader['0']) ? $responseHeader['0'] : '';

        return $httpHeaders;
    }

    private function convertResponse() {
        $responseObj = null;
        if (isset($this->responseHeaders['Content-Type'])) {
            if (preg_match('/^(?:application|text)\/(?:[a-z]+(?:[\.-][0-9a-z]+){0,}[\+\.]|x-)?json(?:-[a-z]+)?/i', $this->responseHeaders['Content-Type'])) {
                $responseObj = json_decode($this->rawResponse, true);
            } elseif (preg_match('~^(?:text/|application/(?:atom\+|rss\+)?)xml~i', $this->responseHeaders['Content-Type'])) {
                $arrayObj = new ArrayUtils();
				        $responseObj = $arrayObj->xmlToArray((string)$this->rawResponse);
            }
        }
        return ($responseObj === null) ? array($this->rawResponse) : $responseObj;
    }

    private function getOperation() {
        $operation = $_REQUEST['route'].'::'.$_REQUEST['endpoint'];
        return $operation;
    }

    public function setCert($cert = false) {
        if ($cert) {
            $this->cert = dirname(__FILE__) . DIRECTORY_SEPARATOR . SystemConfig::CERT_DIR . DIRECTORY_SEPARATOR . SystemConfig::CERT_FILE_NAME;
            $this->setOpt(CURLOPT_CAINFO, $this->cert);
        } else {
            $this->setOpt(CURLOPT_SSL_VERIFYHOST, false);
            $this->setOpt(CURLOPT_SSL_VERIFYPEER, false);
        }
    }
}
